package ru.yandex.autotests.direct.api.adgroups.get;

import com.yandex.direct.api.v5.general.StatusEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsStories;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.autotests.direct.utils.model.RegionIDValues;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Author pavryabov
 * Date 08.10.14
 * https://jira.yandex-team.ru/browse/TESTIRT-2022
 */
@Aqua.Test
@Features(AdGroupsFeatures.GET)
@Stories(AdGroupsStories.STATUSES)
@Tag(TagDictionary.TRUNK)
public class GroupSendToModerateTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(AdGroupsLogins.CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    Long campaignId;
    Long adGroupId;
    Long adId;
    static final String TITLE = "Title";

    @Before
    @Step("Подготовка данных для теста")
    public void createGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdGroupsLogins.CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(AdGroupsLogins.CLIENT);
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(AdGroupsLogins.CLIENT)
                .campaignsSteps().setCampaignsStatusModerate(campaignId, CampaignsStatusmoderate.Yes);
        api.userSteps.campaignFakeSteps().makeCampaignModerated(campaignId);
        //в Before, чтобы баннеры при создании не отправлялись на модерацию
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
        adId = api.userSteps.adsSteps().addAd(
                new AdAddItemMap().withTextAd(
                        new TextAdAddMap().defaultTextAd()
                        .withTitle(TITLE))
                        .withAdGroupId(adGroupId));
    }

    @Test
    public void sendToModerateBannerFromGroup() {
        StatusEnum status = api.userSteps.adGroupsSteps().getGroupStatus(adGroupId);
        assumeThat("группа имеет статус DRAFT", status, equalTo(StatusEnum.DRAFT));
        api.userSteps.adsSteps().adsModerate(adId);
        status = api.userSteps.adGroupsSteps().getGroupStatus(adGroupId);
        assertThat("группа отправилась на модерацию", status, equalTo(StatusEnum.MODERATION));
    }

    @Test
    public void updateRegionIdsInRejectedGroup() {
        //DIRECT-35661
        //DIRECT-35071
        api.userSteps.groupFakeSteps().setStatusModerate(adGroupId, Status.NO);
        StatusEnum status = api.userSteps.adGroupsSteps().getGroupStatus(adGroupId);
        assumeThat("группа имеет статус REJECTED", status, equalTo(StatusEnum.REJECTED));
        api.userSteps.adGroupsSteps().adGroupsUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap()
                                .withId(adGroupId)
                                .withRegionIds(RegionIDValues.KAZAKHSTAN.getId().longValue())
                )
        );
        status = api.userSteps.adGroupsSteps().getGroupStatus(adGroupId);
        assertThat("группа отправилась на модерацию", status, equalTo(StatusEnum.MODERATION));
    }

    @Test
    public void updateNegativeKeywordsInRejectedGroup() {
        //DIRECT-35661
        api.userSteps.groupFakeSteps().setStatusModerate(adGroupId, Status.NO);
        StatusEnum status = api.userSteps.adGroupsSteps().getGroupStatus(adGroupId);
        assumeThat("группа имеет статус REJECTED", status, equalTo(StatusEnum.REJECTED));
        api.userSteps.adGroupsSteps().adGroupsUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap()
                                .withId(adGroupId)
                                .withNegativeKeywords(AdGroupAddItemMap.DEFAULT_NEGATIVE_KEYWORD + "new")
                )
        );
        status = api.userSteps.adGroupsSteps().getGroupStatus(adGroupId);
        assertThat("группа не отправилась на модерацию", status, equalTo(StatusEnum.REJECTED));
    }

    @Test
    public void updateRegionIdsInPreacceptedGroup() {
        //DIRECT-35661
        //DIRECT-35071
        api.userSteps.groupFakeSteps().setStatusModerate(adGroupId, Status.READY);
        api.userSteps.groupFakeSteps().setStatusPostModerate(adGroupId, Status.YES);
        StatusEnum status = api.userSteps.adGroupsSteps().getGroupStatus(adGroupId);
        assumeThat("группа имеет статус PREACCEPTED", status, equalTo(StatusEnum.PREACCEPTED));
        api.userSteps.adGroupsSteps().adGroupsUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap()
                                .withId(adGroupId)
                                .withRegionIds(RegionIDValues.KAZAKHSTAN.getId().longValue())
                )
        );
        status = api.userSteps.adGroupsSteps().getGroupStatus(adGroupId);
        assertThat("группа отправилась на модерацию", status, equalTo(StatusEnum.MODERATION));
    }

    @Test
    public void updateNegativeKeywordsInPreacceptedGroup() {
        //DIRECT-35661
        api.userSteps.groupFakeSteps().setStatusModerate(adGroupId, Status.READY);
        api.userSteps.groupFakeSteps().setStatusPostModerate(adGroupId, Status.YES);
        StatusEnum status = api.userSteps.adGroupsSteps().getGroupStatus(adGroupId);
        assumeThat("группа имеет статус PREACCEPTED", status, equalTo(StatusEnum.PREACCEPTED));
        api.userSteps.adGroupsSteps().adGroupsUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap()
                                .withId(adGroupId)
                                .withNegativeKeywords(AdGroupAddItemMap.DEFAULT_NEGATIVE_KEYWORD + "new")
                )
        );
        status = api.userSteps.adGroupsSteps().getGroupStatus(adGroupId);
        assertThat("группа не отправилась на модерацию", status, equalTo(StatusEnum.PREACCEPTED));
    }

    @Test
    public void updateRegionIdsInAcceptedGroup() {
        //DIRECT-35661
        //DIRECT-35071
        api.userSteps.groupFakeSteps().setStatusModerate(adGroupId, Status.YES);
        api.userSteps.groupFakeSteps().setStatusPostModerate(adGroupId, Status.YES);
        StatusEnum status = api.userSteps.adGroupsSteps().getGroupStatus(adGroupId);
        assumeThat("группа имеет статус ACCEPTED", status, equalTo(StatusEnum.ACCEPTED));
        api.userSteps.adGroupsSteps().adGroupsUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap()
                                .withId(adGroupId)
                                .withRegionIds(RegionIDValues.KAZAKHSTAN.getId().longValue())
                )
        );
        status = api.userSteps.adGroupsSteps().getGroupStatus(adGroupId);
        assertThat("группа отправилась на модерацию", status, equalTo(StatusEnum.MODERATION));
    }

    @Test
    public void updateNegativeKeywordsInAcceptedGroup() {
        //DIRECT-35661
        api.userSteps.groupFakeSteps().setStatusModerate(adGroupId, Status.YES);
        api.userSteps.groupFakeSteps().setStatusPostModerate(adGroupId, Status.YES);
        StatusEnum status = api.userSteps.adGroupsSteps().getGroupStatus(adGroupId);
        assumeThat("группа имеет статус ACCEPTED", status, equalTo(StatusEnum.ACCEPTED));
        api.userSteps.adGroupsSteps().adGroupsUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap()
                                .withId(adGroupId)
                                .withNegativeKeywords(AdGroupAddItemMap.DEFAULT_NEGATIVE_KEYWORD + "new")
                )
        );
        status = api.userSteps.adGroupsSteps().getGroupStatus(adGroupId);
        assertThat("группа не отправилась на модерацию", status, equalTo(StatusEnum.ACCEPTED));
    }

}
