package ru.yandex.autotests.direct.api.adgroups.get;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import com.yandex.direct.api.v5.adgroups.AdGroupFieldEnum;
import com.yandex.direct.api.v5.adgroups.GetResponse;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupGetItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.LimitOffsetMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.extractProperty;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Author pavryabov
 * Date 28.08.14
 * https://jira.yandex-team.ru/browse/TESTIRT-2022
 */
@Aqua.Test
@Features(AdGroupsFeatures.GET)
@Tag(TagDictionary.TRUNK)
public class PagerTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(AdGroupsLogins.CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    static Long campaignId;
    static Long id1;
    static Long id2;
    static Long id3;
    static Long id4;

    @BeforeClass
    public static void initTest() {
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(AdGroupsLogins.CLIENT);
        id1 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        id2 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        id3 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        id4 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
    }

    @Before
    public void clearSpentUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdGroupsLogins.CLIENT);
    }

    @Test
    public void getHead() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(2L)
                                .withOffset(0L))
        );
        List<Long> gotIds = extractProperty(response.getAdGroups(), AdGroupGetItemMap.ID);
        assertThat("вернулся правильный набор групп", gotIds, equalTo(Arrays.asList(id1, id2)));
    }

    @Test
    public void getBody() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(2L)
                                .withOffset(1L))
        );
        List<Long> gotIds = extractProperty(response.getAdGroups(), AdGroupGetItemMap.ID);
        assertThat("вернулся правильный набор групп", gotIds, equalTo(Arrays.asList(id2, id3)));
    }

    @Test
    public void getTail() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(2L)
                                .withOffset(2L))
        );
        List<Long> gotIds = extractProperty(response.getAdGroups(), AdGroupGetItemMap.ID);
        assertThat("вернулся правильный набор групп", gotIds, equalTo(Arrays.asList(id3, id4)));
    }

    @Test
    public void getPartOfTail() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(2L)
                                .withOffset(3L))
        );
        List<Long> gotIds = extractProperty(response.getAdGroups(), AdGroupGetItemMap.ID);
        assertThat("вернулся правильный набор групп", gotIds, equalTo(Collections.singletonList(id4)));
    }

    @Test
    public void getOverTail() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(2L)
                                .withOffset(4L))
        );
        assertThat("вернулся правильный набор групп", response.getAdGroups(), hasSize(0));
    }

    @Test
    public void onlyLimit() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(2L))
        );
        List<Long> gotIds = extractProperty(response.getAdGroups(), AdGroupGetItemMap.ID);
        assertThat("вернулся правильный набор групп", gotIds, equalTo(Arrays.asList(id1, id2)));
    }

    @Test
    public void onlyOffset() {
        //DIRECT-34629
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withOffset(2L))
        );
        List<Long> gotIds = extractProperty(response.getAdGroups(), AdGroupGetItemMap.ID);
        assertThat("вернулся правильный набор групп", gotIds, equalTo(Arrays.asList(id3, id4)));
    }

    @Test
    public void zeroLimit() {
        //DIRECT-34629
        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(0L)),
                new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0)
        );
    }

    @Test
    public void maxLimit() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(LimitOffsetMap.LIMIT_MAX_LENGTH))
        );
        List<Long> gotIds = extractProperty(response.getAdGroups(), AdGroupGetItemMap.ID);
        assertThat("вернулся правильный набор групп", gotIds, equalTo(Arrays.asList(id1, id2, id3, id4)));
    }

    @Test
    public void overMaxLimit() {
        //DIRECT-34627
        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(LimitOffsetMap.LIMIT_MAX_LENGTH + 1)),
                new Api5Error(4002, Api5ErrorDetails.LIMIT_CANNOT_BE_MORE,
                        LimitOffsetMap.LIMIT_MAX_LENGTH)
        );
    }

    @Test
    public void checkLimitedByIfLimitNotWorked() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(4L))
        );
        assertThat("значение LimitedBy верно", response.getLimitedBy(), nullValue());
    }

    @Test
    public void negativeLimit() {
        //DIRECT-34626
        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(-1L)),
                new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0)
        );
    }

    @Test
    public void zeroOffset() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(2L)
                                .withOffset(0L))
        );
        List<Long> gotIds = extractProperty(response.getAdGroups(), AdGroupGetItemMap.ID);
        assertThat("вернулся правильный набор групп", gotIds, equalTo(Arrays.asList(id1, id2)));
    }

    @Test
    public void negativeOffset() {
        //DIRECT-34626
        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withOffset(-1L)),
                new Api5Error(4002,
                        Api5ErrorDetails.OFFSET_MUST_BE_MORE_OR_EQUAL_THAN_0)
        );
    }

    @Test
    public void checkLimitedByIfLimitIsWorked() {
        //DIRECT-34633
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(2L))
        );
        assertThat("значение LimitedBy верное", response.getLimitedBy(), equalTo(2L));
    }

    @Test
    public void checkLimitedByIfLimitIsWorkedWithOffset() {
        //DIRECT-44575
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withCampaignIds(campaignId)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withOffset(1L)
                                .withLimit(2L))
        );
        assertThat("значение LimitedBy верное", response.getLimitedBy(), equalTo(3L));
    }
}
