package ru.yandex.autotests.direct.api.adgroups.get.negativekeywords;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import com.yandex.direct.api.v5.adgroups.AdGroupFieldEnum;
import com.yandex.direct.api.v5.adgroups.GetResponse;
import org.apache.commons.lang3.RandomStringUtils;
import org.hamcrest.Matcher;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.banners.KeywordsSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.Matchers.either;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(AdGroupsFeatures.GET)
@Tag(TagDictionary.TRUNK)
@Issue("https://st.yandex-team.ru/TESTIRT-10419")
@Description("Получение минус-фразы на группу")
@RunWith(Parameterized.class)
public class GetNegativeKeyPhrasesTest {

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(AdGroupsLogins.CLIENT);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @SuppressWarnings("DefaultAnnotationParam")
    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public List<String> negativePhrases;

    @Parameterized.Parameter(2)
    public Matcher<List<String>> negativeKeywordsAfterSaveMatcher;

    @Parameterized.Parameters(name = "{0}")
    public static Collection testData() {
        String maxLengthNegativeWord = RandomStringUtils.randomAlphabetic(KeywordsSteps.MINUS_WORD_MAX_LENGTH);
        String maxWordInNegativePhrase = "белый медведь шьет !себе шубу !на зиму";

        ArrayList<String> maxLengthNegativePhrasesList = new ArrayList<>();
        maxLengthNegativePhrasesList.add(maxWordInNegativePhrase);
        int allPhrasesLength = maxWordInNegativePhrase.length();
        while (allPhrasesLength <= (KeywordsSteps.PHRASE_MAX_LENGTH - KeywordsSteps.MINUS_WORD_MAX_LENGTH)) {
            allPhrasesLength = allPhrasesLength + KeywordsSteps.MINUS_WORD_MAX_LENGTH;
            maxLengthNegativePhrasesList.add(RandomStringUtils.randomAlphabetic(KeywordsSteps.MINUS_WORD_MAX_LENGTH));
        }
        if (KeywordsSteps.PHRASE_MAX_LENGTH - allPhrasesLength > 0) {
            maxLengthNegativePhrasesList.add(
                    RandomStringUtils.randomAlphabetic(KeywordsSteps.MINUS_WORD_MAX_LENGTH)
                            .substring(0, KeywordsSteps.PHRASE_MAX_LENGTH - allPhrasesLength));
        }
        Collections.sort(maxLengthNegativePhrasesList);

        @SuppressWarnings("squid:S1192")
        Object[][] data = new Object[][]{
                {"Min 2 words for key phrase",
                        singletonList("белый медведь"),
                        equalTo(singletonList("белый медведь"))},
                {"Max 7 words for key phrase",
                        singletonList(maxWordInNegativePhrase),
                        equalTo(singletonList("белый медведь шьет !себе шубу !на зиму"))},
                {"Max length for word in negative key phrase",
                        singletonList(maxLengthNegativeWord + " белый медведь"),
                        either(equalTo(singletonList(maxLengthNegativeWord.toLowerCase() + " белый медведь"))) // Perl
                                .or(equalTo(singletonList(maxLengthNegativeWord + " белый медведь")))}, // Java
                {"Max length 4096 symbol for negative key phrase",
                        maxLengthNegativePhrasesList,
                        either(equalTo(maxLengthNegativePhrasesList.stream().map(String::toLowerCase).sorted().collect(toList()))) // Perl
                                .or(equalTo(maxLengthNegativePhrasesList.stream().sorted().collect(toList())))}, // Java
                {"Phrase with a dash",
                        singletonList("онлайн-переводчик"),
                        // TODO DIRECT-75345
                        either(equalTo(singletonList("онлайн-переводчик")))
                                .or(equalTo(singletonList("онлайн переводчик")))},
                {"+в горшках цветы",
                        singletonList("+в горшках цветы"),
                        equalTo(singletonList("+в горшках цветы"))},
                {"цветы +в горшках",
                        singletonList("цветы +в горшках"),
                        equalTo(singletonList("цветы +в горшках"))},
                {"кот в мешке, цветы +в горшках",
                        asList("кот в мешке", "цветы +в горшках"),
                        equalTo(asList("кот !в мешке", "цветы +в горшках"))},
                {"купить !цветы",
                        singletonList("купить !цветы"),
                        equalTo(singletonList("купить !цветы"))},
                {"!купить !цветы",
                        singletonList("!купить !цветы"),
                        equalTo(singletonList("!купить !цветы"))},
                {"\"цветы в горшках\"",
                        singletonList("\"цветы в горшках\""),
                        equalTo(singletonList("\"цветы в горшках\""))},
                {"\"купить цветы +в горшках недорого\"",
                        singletonList("\"купить цветы +в горшках недорого\""),
                        equalTo(singletonList("\"купить цветы +в горшках недорого\""))},
                {"[из Москвы в Париж]",
                        singletonList("[из Москвы в Париж]"),
                        either(equalTo(singletonList("[из москвы в париж]"))) // Perl
                                .or(equalTo(singletonList("[из Москвы в Париж]")))}, // Java
                {"2 билета [из Москвы в Париж]",
                        singletonList("2 билета [из Москвы в Париж]"),
                        either(equalTo(singletonList("2 билета [из москвы в париж]"))) // Perl
                                .or(equalTo(singletonList("2 билета [из Москвы в Париж]")))}, // Java
                {"билеты [из Москвы в Париж]",
                        singletonList("билеты [из Москвы в Париж]"),
                        either(equalTo(singletonList("билеты [из москвы в париж]")))  // Perl
                                .or(equalTo(singletonList("билеты [из Москвы в Париж]")))}, // Java
                {"[из !Москвы в Париж]",
                        singletonList("[из !Москвы в Париж]"),
                        either(equalTo(singletonList("[из !москвы в париж]"))) // Perl
                                .or(equalTo(singletonList("[из !Москвы в Париж]")))}, // Java
                {" из !Москвы в Париж",
                        singletonList(" из !Москвы в Париж"),
                        either(equalTo(singletonList("!из !москвы !в париж"))) // Perl
                                .or(equalTo(singletonList("!из !Москвы !в Париж")))}, // Java
                {"\"[из Москвы в Париж]\"",
                        singletonList("\"[из Москвы в Париж]\""),
                        either(equalTo(singletonList("\"[из москвы в париж]\""))) // Perl
                                .or(equalTo(singletonList("\"[из Москвы в Париж]\"")))}, // Java
                {"говорить +по-английски",
                        singletonList("говорить +по-английски"),
                        // TODO DIRECT-75345
                        either(equalTo(singletonList("говорить +по-английски")))
                            .or(equalTo(singletonList("говорить +по +английски")))},
                {"[англо-русский переводчик]",
                        singletonList("[англо-русский переводчик]"),
                        // TODO DIRECT-75345
                        either(equalTo(singletonList("[англо-русский переводчик]")))
                            .or(equalTo(singletonList("[англо русский переводчик]")))},
        };
        return asList(data);
    }

    private static AdGroupAddItemMap adGroupAddItemMap;
    private Long id;

    @BeforeClass
    public static void createCampaign() {
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        adGroupAddItemMap = new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdGroupsLogins.CLIENT);
        id = api.userSteps.adGroupsSteps().addGroup(adGroupAddItemMap.withNegativeKeywords(negativePhrases));
    }

    @Test
    public void getGroupWithNegativeKeyPhrases() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withIds(id)
                        )
                        .withFieldNames(AdGroupFieldEnum.NEGATIVE_KEYWORDS)
        );
        assumeThat("AdGroups.get вернул одну группу", response.getAdGroups(), hasSize(1));
        assertThat("минус-фразы вернулись правильно",
                response.getAdGroups().get(0).getNegativeKeywords().getValue().getItems(),
                negativeKeywordsAfterSaveMatcher);
    }
}
