package ru.yandex.autotests.direct.api.adgroups.get.restrictedregions;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.adgroups.AdGroupFieldEnum;
import com.yandex.direct.api.v5.adgroups.AdGroupGetItem;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersMinusGeoType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusarch;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusshow;
import ru.yandex.autotests.direct.utils.model.RegionIDValues;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.CampaignsType;
import ru.yandex.autotests.directapi.enums.AdGroupType;
import ru.yandex.autotests.directapi.enums.BannerType;
import ru.yandex.autotests.directapi.enums.CampaignType;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static com.google.common.primitives.Longs.asList;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

@Aqua.Test
@Features(AdGroupsFeatures.GET)
@RunWith(Parameterized.class)
@Description("Проверка получения RestrictedRegionIds для групп разных типов")
public class GetRestrictedRegionIdsTest {

    private static final Long DEFAULT_REGION_ID = 0L;
    private static final Long RUSSIA = RegionIDValues.RUSSIA_COUNTRY.getId().longValue();
    private static final Long MOSCOW_AND_MOSCOW_REGION = RegionIDValues.MOSCOW_AND_MOSCOW_REGION.getId().longValue();
    private static final Long CRIMEA = RegionIDValues.CRIMEA.getId().longValue();
    private static final Long KAZAKHSTAN = RegionIDValues.KAZAKHSTAN.getId().longValue();
    private static final Long BELARUS = RegionIDValues.BELARUS.getId().longValue();
    private static final Long TURKEY = RegionIDValues.TURKEY.getId().longValue();
    private static final Long UKRAINE = RegionIDValues.UKRAINE.getId().longValue();
    private static final Long KIEV = RegionIDValues.KIEV.getId().longValue();
    private static final Long SPB = RegionIDValues.SPB.getId().longValue();

    private static final String clientLogin = AdGroupsLogins.CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(clientLogin);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter()
    public CampaignType campaignType;

    @Parameterized.Parameter(1)
    public AdGroupType adGroupType;

    @Parameterized.Parameter(2)
    public BannerType adType;

    private Long campaignId;

    @Parameterized.Parameters(name = "campaign type [{0}], group type [{1}], banner type [{2}]")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {CampaignType.TEXT, AdGroupType.BASE, BannerType.TEXT},
                {CampaignType.TEXT, AdGroupType.BASE, BannerType.IMAGE_AD},
                {CampaignType.MOBILE_CONTENT, AdGroupType.MOBILE_CONTENT, BannerType.MOBILE_CONTENT},
                {CampaignType.MOBILE_CONTENT, AdGroupType.MOBILE_CONTENT, BannerType.IMAGE_AD},
                {CampaignType.DYNAMIC, AdGroupType.DYNAMIC, BannerType.DYNAMIC},
                {CampaignType.CPM_BANNER, AdGroupType.CPM_BANNER, BannerType.CPM_BANNER},
        });
    }

    @Before
    @Step("Подготовка данных для всех тестов")
    public void before() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(clientLogin);
        campaignId = api.userSteps.campaignSteps().addDefaultCampaign(campaignType);
        if (campaignType == CampaignType.CPM_BANNER) {
            api.userSteps.campaignFakeSteps().setType(campaignId, CampaignsType.CPM_BANNER);
        }
    }

    @Test
    @Description("У объявлений группы нет минус-регионов - RestrictedRegionIds пуст")
    public void noRestrictedRegionsTest() {
        Long adGroupId = api.userSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId).withDefaultGroup(adGroupType)
                        .withRegionIds(DEFAULT_REGION_ID));

        AdGroupGetItem adGroup =
                api.userSteps.adGroupsSteps().getAdGroup(adGroupId, AdGroupFieldEnum.RESTRICTED_REGION_IDS);
        assertThat("вернулись ожидаемые регионы", adGroup.getRestrictedRegionIds().getValue(), nullValue());
    }

    @Test
    @Description("Минус-регион группы не включается в RestrictedRegionIds")
    public void adGroupMinusRegionsAreNotIncludedInRestrictedRegionIds() {
        Long adGroupId = api.userSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId).withDefaultGroup(adGroupType)
                        .withRegionIds(DEFAULT_REGION_ID, -UKRAINE, -BELARUS));

        AdGroupGetItem adGroup =
                api.userSteps.adGroupsSteps().getAdGroup(adGroupId, AdGroupFieldEnum.RESTRICTED_REGION_IDS);
        assertThat("вернулись ожидаемые регионы", adGroup.getRestrictedRegionIds().getValue(), nullValue());
    }

    @Test
    @Description("RestrictedRegionIds есть объединение минус регионов объявлений без дублей")
    public void adGroupRestrictedRegionIdsIsUnionOfAdsMinusRegionsWithouDoubles() {
        Long adGroupId = api.userSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId).withDefaultGroup(adGroupType)
                        .withRegionIds(DEFAULT_REGION_ID));

        createAdWithMinusRegions(adGroupId, new Long[]{UKRAINE, TURKEY}, BannersMinusGeoType.current);
        createAdWithMinusRegions(adGroupId, new Long[]{KAZAKHSTAN, TURKEY}, BannersMinusGeoType.current);

        AdGroupGetItem adGroup =
                api.userSteps.adGroupsSteps().getAdGroup(adGroupId, AdGroupFieldEnum.RESTRICTED_REGION_IDS);
        assertThat("вернулись ожидаемые регионы", adGroup.getRestrictedRegionIds().getValue().getItems(),
                containsInAnyOrder(UKRAINE, TURKEY, KAZAKHSTAN));
    }

    @Test
    @Description("Mинус-регион объявления не включается в RestrictedRegionIds если у группы есть такой же минус-регион")
    public void adsMinusRegionIsExcludedFromRestrictedRegionIdsIfEqualsToOneOfAdGroupMinusRegions() {
        Long adGroupId = api.userSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId).withDefaultGroup(adGroupType)
                        .withRegionIds(DEFAULT_REGION_ID, -CRIMEA));

        createAdWithMinusRegions(adGroupId, new Long[]{UKRAINE, TURKEY, CRIMEA}, BannersMinusGeoType.current);

        AdGroupGetItem adGroup =
                api.userSteps.adGroupsSteps().getAdGroup(adGroupId, AdGroupFieldEnum.RESTRICTED_REGION_IDS);
        assertThat("вернулись ожидаемые регионы", adGroup.getRestrictedRegionIds().getValue().getItems(),
                containsInAnyOrder(UKRAINE, TURKEY));
    }

    @Test
    @Description("Mинус-регион объявления включается в RestrictedRegionIds если он является родительским по отношению к минус-региону группы")
    public void adsMinusRegionIncludedInRestrictedRegionIdsIfItParentToOneOfAdGroupMinusRegions() {
        Long adGroupId = api.userSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId).withDefaultGroup(adGroupType)
                        .withRegionIds(RUSSIA, -MOSCOW_AND_MOSCOW_REGION, KAZAKHSTAN));

        createAdWithMinusRegions(adGroupId, new Long[]{RUSSIA, UKRAINE}, BannersMinusGeoType.current);

        AdGroupGetItem adGroup =
                api.userSteps.adGroupsSteps().getAdGroup(adGroupId, AdGroupFieldEnum.RESTRICTED_REGION_IDS);
        assertThat("вернулись ожидаемые регионы", adGroup.getRestrictedRegionIds().getValue().getItems(),
                containsInAnyOrder(RUSSIA));
    }

    @Test
    public void adsMinusRegionIsExcludedFromRestrictedRegionIdsIfAdIsHidden() {
        Long adGroupId = api.userSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId).withDefaultGroup(adGroupType)
                        .withRegionIds(DEFAULT_REGION_ID));

        Long hiddenAdId = createAdWithMinusRegions(adGroupId, new Long[]{KAZAKHSTAN}, BannersMinusGeoType.current);

        api.userSteps.getDirectJooqDbSteps().useShardForLogin(clientLogin).bannersSteps()
                .setBannerStatusShow(hiddenAdId, BannersStatusshow.No);

        AdGroupGetItem adGroup =
                api.userSteps.adGroupsSteps().getAdGroup(adGroupId, AdGroupFieldEnum.RESTRICTED_REGION_IDS);

        assertThat("Минус регионы невидимых баннеров не должны быть запрещены для группы",
                adGroup.getRestrictedRegionIds().getValue(), nullValue());
    }

    @Test
    public void adsMinusRegionIsExcludedFromRestrictedRegionIdsIfAdIsArchived() {
        Long adGroupId = api.userSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId).withDefaultGroup(adGroupType)
                        .withRegionIds(DEFAULT_REGION_ID));

        Long archivedAdId = createAdWithMinusRegions(adGroupId, new Long[]{UKRAINE}, BannersMinusGeoType.current);

        api.userSteps.getDirectJooqDbSteps().useShardForLogin(clientLogin).bannersSteps()
                .setBannerStatusArchived(archivedAdId, BannersStatusarch.Yes);

        AdGroupGetItem adGroup =
                api.userSteps.adGroupsSteps().getAdGroup(adGroupId, AdGroupFieldEnum.RESTRICTED_REGION_IDS);

        assertThat("Минус регионы архивных баннеров не должны быть запрещены для группы",
                adGroup.getRestrictedRegionIds().getValue(), nullValue());
    }

    @Test
    public void adsMinusRegionIsExcludedFromRestrictedRegionIdsIfAdIsRejected() {
        Long adGroupId = api.userSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId).withDefaultGroup(adGroupType)
                        .withRegionIds(DEFAULT_REGION_ID));

        Long rejectedAdId = createAdWithMinusRegions(adGroupId, new Long[]{BELARUS}, BannersMinusGeoType.current);

        api.userSteps.getDirectJooqDbSteps().useShardForLogin(clientLogin).bannersSteps()
                .setBannerStatusPostModerate(rejectedAdId, BannersStatuspostmoderate.Rejected);

        AdGroupGetItem adGroup =
                api.userSteps.adGroupsSteps().getAdGroup(adGroupId, AdGroupFieldEnum.RESTRICTED_REGION_IDS);

        assertThat("Минус регионы отклоненных баннеров не должны быть запрещены для группы",
                adGroup.getRestrictedRegionIds().getValue(), nullValue());
    }

    @Test
    public void adsMinusRegionIsExcludedFromRestrictedRegionIdsIfMinusRegionsTypeIsNotCurrent() {
        Long adGroupId = api.userSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId).withDefaultGroup(adGroupType)
                        .withRegionIds(DEFAULT_REGION_ID));

        createAdWithMinusRegions(adGroupId, new Long[]{BELARUS}, BannersMinusGeoType.bs_synced);

        AdGroupGetItem adGroup =
                api.userSteps.adGroupsSteps().getAdGroup(adGroupId, AdGroupFieldEnum.RESTRICTED_REGION_IDS);

        assertThat("Минус регионы c типом bs_synced не должны быть запрещены для группы",
                adGroup.getRestrictedRegionIds().getValue(), nullValue());
    }

    @Test
    public void correctRestrictedRegionIds() {
        Long adGroupId = api.userSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId).withDefaultGroup(adGroupType)
                        .withRegionIds(RUSSIA, -MOSCOW_AND_MOSCOW_REGION, UKRAINE, -KIEV, BELARUS));

        createAdWithMinusRegions(adGroupId, new Long[]{SPB, UKRAINE, BELARUS}, BannersMinusGeoType.current);

        AdGroupGetItem adGroup =
                api.userSteps.adGroupsSteps().getAdGroup(adGroupId, AdGroupFieldEnum.RESTRICTED_REGION_IDS);

        assertThat("Вернулись корректные регионы",
                adGroup.getRestrictedRegionIds().getValue().getItems(), is(asList(UKRAINE, BELARUS, RUSSIA)));
    }

    @Test
    public void restrictedRegionIdsDoNotContainMinusIds() {
        Long adGroupId = api.userSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId).withDefaultGroup(adGroupType)
                        .withRegionIds(RUSSIA, -MOSCOW_AND_MOSCOW_REGION));

        createAdWithMinusRegions(adGroupId, new Long[]{RUSSIA}, BannersMinusGeoType.current);

        AdGroupGetItem adGroup =
                api.userSteps.adGroupsSteps().getAdGroup(adGroupId, AdGroupFieldEnum.RESTRICTED_REGION_IDS);

        assertThat("Вернулись корректные регионы",
                adGroup.getRestrictedRegionIds().getValue().getItems(), is(Collections.singletonList(RUSSIA)));
    }

    private Long createAdWithMinusRegions(Long adGroupId, Long[] adMinusRegionIds, BannersMinusGeoType minusGeoType) {
        Long adId;
        if (adType.equals(BannerType.IMAGE_AD)) {
            String imageHash = api.userSteps.imagesStepsV5().addDefaultImageAdAdImage();
            adId = api.userSteps.adsSteps()
                    .addAdWithAdImageHash(adGroupId, adGroupType, BannerType.IMAGE_AD, imageHash);
        } else if (adType.equals(BannerType.CPM_BANNER)) {
            Long creativeId = api.userSteps.getDirectJooqDbSteps().useShardForLogin(clientLogin).perfCreativesSteps()
                    .saveDefaultCanvasCreativesForClient(Long.parseLong(User.get(clientLogin).getClientID()));
            adId = api.userSteps.adsSteps().addDefaultCpmBannerAdBuilderAd(adGroupId, creativeId);
        } else {
            adId = api.userSteps.adsSteps().addDefaultAd(adGroupId, adType);
        }

        String minusRegionIds =
                String.join(",", Arrays.stream(adMinusRegionIds).map(Object::toString).collect(Collectors.toList()));

        api.userSteps.getDirectJooqDbSteps().useShardForLogin(clientLogin).bannersSteps()
                .saveBannersMinusGeo(adId, minusGeoType, minusRegionIds);

        return adId;
    }
}
