package ru.yandex.autotests.direct.api.adgroups.update;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.GroupFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.common.RegionIDValues;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by zakhar on 09.03.16.
 * https://st.yandex-team.ru/TESTIRT-8657
 * DIRECT-35080
 */
@Aqua.Test
@Features(AdGroupsFeatures.UPDATE)
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class UpdateGeoAndCheckStatusBsSyncedTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(AdGroupsLogins.CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    Long adGroupId;
    Long adId;

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public String geoBeforeUpdate;

    @Parameterized.Parameter(2)
    public String geoAfterUpdate;

    @Parameterized.Parameter(3)
    public String expectedGroupStatusBsSynced;

    @Parameterized.Parameter(4)
    public String expectedBannerStatusBsSynced;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> testData() {
        Object[][] data = new Object[][]{
                {"Moscow -> Spb",
                        RegionIDValues.MOSCOW_ID, RegionIDValues.SPB_ID,
                        Status.NO, Status.YES},
                {"Moscow -> Kazakhstan",
                        RegionIDValues.MOSCOW_ID, RegionIDValues.KAZAKH_REGION_ID,
                        Status.NO, Status.NO},
                {"Kazakhstan -> Moscow",
                        RegionIDValues.KAZAKH_REGION_ID, RegionIDValues.MOSCOW_ID,
                        Status.NO, Status.NO},
                // при изменение гео группы, если регион широкий (не город), то есть geoflag для баннеров не выставлен,
                // баннер не должен переотравлятся в БК
                // https://wiki.yandex-team.ru/users/pavryabov/statusbssynced/#gruppy
                {"Moscow region -> Leningrad region",
                        RegionIDValues.MOSCOW_AND_MOSCOW_REGION_ID, RegionIDValues.SPB_AND_LENINGRAD_REGION_ID,
                        Status.NO, Status.YES},

        };
        return Arrays.asList(data);
    }

    private String getGroupStatusBsSynced(Long adGroupID) {
        GroupFakeInfo groupFakeInfo = api.userSteps.groupFakeSteps().getGroupParams(adGroupID);
        return groupFakeInfo.getStatusBsSynced();
    }

    private String getBannerStatusBsSynced(Long bannerID) {
        BannerFakeInfo bannerFakeInfo = api.userSteps.bannersFakeSteps().getBannerParams(bannerID);
        return bannerFakeInfo.getStatusBsSynced();
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createGroupAndBanner() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdGroupsLogins.CLIENT);
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();

        adGroupId = api.userSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                        .withRegionIds(Long.parseLong(geoBeforeUpdate)));

        adId = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);

        api.userSteps.campaignFakeSteps().makeCampaignModerated(campaignId.intValue());

        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(adGroupId, Status.YES);
        api.userSteps.groupFakeSteps().setStatusModerate(adGroupId, Status.YES);

        api.userSteps.bannersFakeSteps().makeBannersModerated(adId);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adId, Status.YES);

        assumeThat("группе установился нужный статус синхронизации с БК",
                getGroupStatusBsSynced(adGroupId), equalTo(Status.YES));
        assumeThat("баннеру установился нужный статус синхронизации с БК",
                getBannerStatusBsSynced(adId), equalTo(Status.YES));
    }

    @Test
    public void updateGeo() {
        api.userSteps.adGroupsSteps().adGroupsUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap().withId(adGroupId)
                                .withRegionIds(Long.parseLong(geoAfterUpdate))));
        assertThat("группа имеет правильный статус синхронизации с БК",
                getGroupStatusBsSynced(adGroupId),
                equalTo(expectedGroupStatusBsSynced));
        assertThat("баннер имеет правильный статус синхронизации с БК",
                getBannerStatusBsSynced(adId),
                equalTo(expectedBannerStatusBsSynced));
    }

}
