package ru.yandex.autotests.direct.api.adgroups.update;

import org.apache.commons.lang3.RandomStringUtils;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.PhrasesRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;

@Aqua.Test
@Features(AdGroupsFeatures.UPDATE)
@Stories(ApiStories.GROUPS)
public class UpdateLongNameTest {
    private static final String client = AdGroupsLogins.CLIENT;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().protocol(ProtocolType.JSON).as(client);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static Long campaignId;
    private static String name;
    private static DirectJooqDbSteps jooqDbSteps;

    @Rule
    public Trashman trashman = new Trashman(api);
    private Long adGroupId;

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdGroupsLogins.CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(AdGroupsLogins.CLIENT);
        name = RandomStringUtils.randomAlphabetic(AdGroupAddItemMap.MAX_NAME_LENGTH * 2);
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps();
        jooqDbSteps.useShardForLogin(client);
    }

    @Before
    public void clearSpentUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdGroupsLogins.CLIENT);
    }

    @Before
    public void longName() {
        //DIRECT-74198
        final AdGroupAddItemMap group = new AdGroupAddItemMap()
                .withCampaignId(campaignId)
                .withName(RandomStringUtils.randomAlphabetic(AdGroupAddItemMap.MAX_NAME_LENGTH / 2))
                .withRegionIds(AdGroupAddItemMap.DEFAULT_REGION_ID);
        adGroupId = api.userSteps.adGroupsSteps().addGroup(group);

        final PhrasesRecord record = jooqDbSteps.adGroupsSteps().getPhrases(adGroupId);
        record.setGroupName(name);
        jooqDbSteps.adGroupsSteps().updatePhrases(record);
    }

    @Test
    public void updateIfLongName() {
        api.userSteps.adGroupsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap()
                                .withId(adGroupId)
                                .withName(name)
                                .withRegionIds(AdGroupAddItemMap.DEFAULT_REGION_ID)),
                ExpectedResult.errors(new Notification(5001,
                        Api5ErrorDetailsJava.FIELD_MAXIMUM_LENGTH_EXCEEDED,
                        capitalize(AdGroupAddItemMap.NAME),
                        AdGroupAddItemMap.MAX_NAME_LENGTH)));
    }

    @Test
    public void updateWithoutNameSuccess() {
        api.userSteps.adGroupsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap()
                                .withId(adGroupId)
                                .withRegionIds(AdGroupAddItemMap.DEFAULT_REGION_ID)),
                ExpectedResult.success(adGroupId));
    }

    @Test
    public void updateWithNormalLengthName() {
        api.userSteps.adGroupsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap()
                                .withId(adGroupId)
                                .withName("short_name")
                ),
                ExpectedResult.success()
        );
    }
}
