package ru.yandex.autotests.direct.api.adgroups.update;

import java.util.Arrays;
import java.util.Collection;
import java.util.function.Function;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.direct.utils.model.RegionIDValues;
import ru.yandex.autotests.direct.utils.model.RegionNameValues;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

/**
 * Author pavryabov
 * Date 18.08.14
 * https://st.yandex-team.ru/TESTIRT-2022
 */
@Aqua.Test
@Features(AdGroupsFeatures.UPDATE)
@RunWith(Parameterized.class)
public class UpdateRegionIdsTest {

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(AdGroupsLogins.CLIENT);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final Long MOSCOW = RegionIDValues.MOSCOW.getId().longValue();
    private static final Long CHERNOGOLOVKA = RegionIDValues.CHERNOGOLOVKA.getId().longValue();
    private static final Long MOSCOW_REGION = RegionIDValues.MOSCOW_AND_MOSCOW_REGION.getId().longValue();
    private static final Long SPB = RegionIDValues.SPB.getId().longValue();
    private static final Long RUSSIA = RegionIDValues.RUSSIA_COUNTRY.getId().longValue();
    private static final Long CRIMEA = RegionIDValues.CRIMEA.getId().longValue();
    private static final Long UKRAINE = RegionIDValues.UKRAINE.getId().longValue();
    private static final Long SEVASTOPOL = RegionIDValues.SEVASTOPOL.getId().longValue();
    private static final Long EUROPE = RegionIDValues.EUROPE.getId().longValue();
    private static final Long SNG = RegionIDValues.SNG.getId().longValue();

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public Long[] regionIds;

    @Parameterized.Parameter(2)
    public ExpectedResult expectedResult;

    private static ExpectedResult expectedResult(
            Api5ErrorDetailsJava javaErrorDetails,
            String... regions)
    {
        String regionList = String.join(", ", regions);
        return ExpectedResult.errors(new Notification(5120, javaErrorDetails, regionList));
    }

    private static ExpectedResult expectedResult(
            Api5ErrorDetailsJava javaErrorDetails,
            Long regionId)
    {
        return ExpectedResult.errors(new Notification(5120, javaErrorDetails, regionId));
    }

    @Parameterized.Parameters(name = "{0}")
    public static Collection testData() {
        String locale = DirectTestRunProperties.getInstance().getDirectApiLocale();
        Function<RegionNameValues, String> localRegionNameGetter = locale.equals("ru")
                ? RegionNameValues::getRussianName
                : RegionNameValues::getEnglishName;

        Object[][] data = new Object[][]{
                {"All regions", new Long[]{0L}, ExpectedResult.success()},
                {"One region", new Long[]{MOSCOW}, ExpectedResult.success()},
                {"Two different regions", new Long[]{MOSCOW, CHERNOGOLOVKA}, ExpectedResult.success()},
                {"Child with parent", new Long[]{MOSCOW, MOSCOW_REGION}, ExpectedResult.success()},
                {"Region with zero", new Long[]{MOSCOW, 0L}, ExpectedResult.success()},
                {"Exclude child from parent", new Long[]{MOSCOW_REGION, -MOSCOW}, ExpectedResult.success()},
                {"Exclude child and child of child", new Long[]{-MOSCOW, -MOSCOW_REGION, RUSSIA},
                        ExpectedResult.success()},
                {"Zero with negative", new Long[]{0L, MOSCOW_REGION, -MOSCOW}, ExpectedResult.success()},
                {"Duplicate region", new Long[]{MOSCOW, CHERNOGOLOVKA, MOSCOW},
                        expectedResult(
                                Api5ErrorDetailsJava.REGION_IS_DULICATED,
                                localRegionNameGetter.apply(RegionNameValues.MOSCOW))
                },
                {"Only negative", new Long[]{-MOSCOW},
                        expectedResult(
                                Api5ErrorDetailsJava.ALL_REGIONS_CANNOT_BE_EXCLUDED,
                                localRegionNameGetter.apply(RegionNameValues.MOSCOW))
                },
                {"Only negatives", new Long[]{-MOSCOW, -CHERNOGOLOVKA},
                        expectedResult(
                                Api5ErrorDetailsJava.ALL_REGIONS_CANNOT_BE_EXCLUDED,
                                localRegionNameGetter.apply(RegionNameValues.MOSCOW),
                                localRegionNameGetter.apply(RegionNameValues.CHERNOGOLOVKA))},
                {"Invalid region", new Long[]{123456L},
                        expectedResult(
                                Api5ErrorDetailsJava.INVALID_OR_NONEXISTENT_REGION,
                                123456L)},
                {"Crimea", new Long[]{CRIMEA}, ExpectedResult.success()},
                {"Exclude from Crimea", new Long[]{CRIMEA, -SEVASTOPOL}, ExpectedResult.success()},
                {"Crimea and Ukraine", new Long[]{CRIMEA, UKRAINE}, ExpectedResult.success()},
                {"Crimea and Russia", new Long[]{CRIMEA, RUSSIA}, ExpectedResult.success()},
                {"Exclude Crimea from Ukraine", new Long[]{UKRAINE, -CRIMEA},
                        expectedResult(
                                Api5ErrorDetailsJava.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.CRIMEA))},
                {"Exclude Crimea from Russia", new Long[]{RUSSIA, -CRIMEA},
                        expectedResult(
                                Api5ErrorDetailsJava.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.CRIMEA))},
                {"Exclude Crimea from Europe", new Long[]{EUROPE, -CRIMEA},
                        expectedResult(
                                Api5ErrorDetailsJava.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.CRIMEA))},
                {"Exclude Crimea from SNG and Ukraine", new Long[]{SNG, UKRAINE, -CRIMEA},
                        expectedResult(
                                Api5ErrorDetailsJava.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.CRIMEA))},
                {"Exclude parent from child", new Long[]{MOSCOW, -MOSCOW_REGION},
                        expectedResult(
                                Api5ErrorDetailsJava.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.MOSCOW_AND_MOSCOW_REGION))},
                {"Exclude other region", new Long[]{MOSCOW, -SPB},
                        expectedResult(
                                Api5ErrorDetailsJava.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.SAINT_PETERSBURG))},
        };
        return Arrays.asList(data);
    }

    static Long campaignId;
    Long id;

    @BeforeClass
    public static void createCampaign() {
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(AdGroupsLogins.CLIENT);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdGroupsLogins.CLIENT);
        id = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
    }

    @Test
    public void updateGroupWithRegionIds() {
        ////DIRECT-34451, DIRECT-34452, DIRECT-34453, DIRECT-35721, DIRECT-35929
        api.userSteps.adGroupsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap()
                                .withId(id)
                                .withRegionIds(regionIds)
                ),
                expectedResult
        );
    }
}
