package ru.yandex.autotests.direct.api.adgroups.update;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.yandex.direct.api.v5.adgroups.AdGroupFieldEnum;
import com.yandex.direct.api.v5.adgroups.GetResponse;
import com.yandex.direct.api.v5.general.ActionResult;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ArrayOfStringMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins.CLIENT;
import static ru.yandex.autotests.direct.utils.matchers.BeanEquals.beanEquals;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;


/**
 * Author pavryabov
 * Date 10.08.14
 * https://jira.yandex-team.ru/browse/TESTIRT-2022
 */
@Aqua.Test
@Features(AdGroupsFeatures.UPDATE)
@Tag(TagDictionary.TRUNK)
public class UpdateTest {

    private static final String PARAMS = "params";
    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);
    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();
    static Long campaignId;
    @Rule
    public Trashman trasher = new Trashman(api);
    Long id;

    @BeforeClass
    public static void initTest() {
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(CLIENT);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        api.as(CLIENT);
        id = api.userSteps.adGroupsSteps().addGroup(new AdGroupAddItemMap()
                .defaultAdGroupAddItem(campaignId)
                .withNegativeKeywords(AdGroupAddItemMap.DEFAULT_NEGATIVE_KEYWORD)
        );
    }

    @Test
    public void updateWithEmptyArrayOfNegativeKeywords() {
        //DIRECT-33530
        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap()
                                .withId(id)
                                .withNegativeKeywords(new String[]{})
                ),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_NOT_LESS_ELEMENTS,
                        path(UpdateRequestMap.AD_GROUPS, AdGroupUpdateItemMap.NEGATIVE_KEYWORDS,
                                ArrayOfStringMap.ITEMS),
                        1));
    }

    @Test
    public void updateWithEmptyRequest() {
        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsUpdate(
                new UpdateRequestMap(),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        path(UpdateRequestMap.AD_GROUPS)));
    }

    @Test
    public void updateTwoGroups() {
        Long id2 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        api.userSteps.adGroupsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap().withId(id).withName(AdGroupAddItemMap.DEFAULT_NAME + "new"),
                        new AdGroupUpdateItemMap().withId(id2).withRegionIds(AdGroupAddItemMap.DEFAULT_REGION_ID + 1)),
                ExpectedResult.success(id),
                ExpectedResult.success(id2)
        );
    }

    @Test
    public void update1000Groups() {
        AdGroupUpdateItemMap[] adGroupUpdateItemMaps = new AdGroupUpdateItemMap[AdGroupUpdateItemMap.GROUPS_MAX_LENGTH];
        Arrays.fill(adGroupUpdateItemMaps, new AdGroupUpdateItemMap().withId(id));
        List<ActionResult> results =
                api.userSteps.adGroupsSteps().adGroupsUpdate(new UpdateRequestMap().withAdGroups(adGroupUpdateItemMaps)
                ).getUpdateResults();
        assertThat("количество объектов в ответе верное",
                results.size(), equalTo(AdGroupUpdateItemMap.GROUPS_MAX_LENGTH));
        Set<ActionResult> setOfResults = new HashSet<>(results);
        assertThat("в каждом объекте ответа вернулась одна и та же ошибка", setOfResults.size(), equalTo(1));
        ExpectedResult expectedResult = ExpectedResult.errors(new Notification(9800));
        assertThat("в каждом объекте ответа вернлась верная ошибка",
                results.get(0), beanEquals((ActionResult) expectedResult.getBean())
                        .accordingStrategy(expectedResult.getActionResultCompareStrategy()));
    }

    @Test
    public void update1001Groups() {
        //DIRECT-34753
        AdGroupUpdateItemMap[] adGroupUpdateItemMaps =
                new AdGroupUpdateItemMap[AdGroupUpdateItemMap.GROUPS_MAX_LENGTH + 1];
        Arrays.fill(adGroupUpdateItemMaps, new AdGroupUpdateItemMap().withId(id));
        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsUpdate(
                new UpdateRequestMap().withAdGroups(adGroupUpdateItemMaps),
                new Api5Error(9300,
                        Api5ErrorDetails.UPDATE_NO_MORE_THAN, AdGroupUpdateItemMap.GROUPS_MAX_LENGTH)
        );
    }

    @Test
    public void updateTwoIvalidGroups() {
        api.userSteps.adGroupsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap().withId(id + 10000),
                        new AdGroupUpdateItemMap().withId(id + 20000)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.GROUP_NOT_FOUND)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.GROUP_NOT_FOUND))
        );
    }

    @Test
    public void updateOneIvalidOneValidGroups() {
        api.userSteps.adGroupsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap().withId(id + 10000),
                        new AdGroupUpdateItemMap().withId(id)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.GROUP_NOT_FOUND)),
                ExpectedResult.success(id)
        );
    }

    @Test
    public void updateGroupTwoTimesInOneRequest() {
        //DIRECT-33205
        api.userSteps.adGroupsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap().withId(id),
                        new AdGroupUpdateItemMap().withId(id)),
                ExpectedResult.errors(new Notification(9800)),
                ExpectedResult.errors(new Notification(9800))
        );
    }

    @Test
    public void updateWithNullNegativeKeywords() {
        //DIRECT-34717
        //DIRECT-34193
        api.userSteps.adGroupsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap()
                                .withId(id)
                                .withNegativeKeywords((ArrayOfStringMap) null)
                ),
                ExpectedResult.success()
        );
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withIds(id)
                        )
                        .withFieldNames(AdGroupFieldEnum.NEGATIVE_KEYWORDS)
        );
        assumeThat("AdGroups.get вернул одну группу", response.getAdGroups(), hasSize(1));
        assumeThat("в ответе вернулся непустой объект NegativeKeywrods",
                response.getAdGroups().get(0).getNegativeKeywords(), notNullValue());
        assertThat("в ответе вернулся nillable тег NegativeKeywrods",
                response.getAdGroups().get(0).getNegativeKeywords().getValue(), nullValue());
    }
}
