package ru.yandex.autotests.direct.api.adgroups.update.restrictedregions;

import java.util.Arrays;
import java.util.Collection;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersMinusGeoType;
import ru.yandex.autotests.direct.utils.model.RegionIDValues;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.darkside.model.CampaignsType;
import ru.yandex.autotests.directapi.enums.AdGroupType;
import ru.yandex.autotests.directapi.enums.BannerType;
import ru.yandex.autotests.directapi.enums.CampaignType;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

@Aqua.Test
@Features(AdGroupsFeatures.UPDATE)
@RunWith(Parameterized.class)
@Description("При обновлении RegionIds должно быть предупреждение в случае пересечения геотаргетинга группы с минус-регионами объявлений")
public class CheckWarningsOnRestrictedRegionsTest {

    private static final String clientLogin = AdGroupsLogins.CLIENT;

    private static final Long RUSSIA = RegionIDValues.RUSSIA_COUNTRY.getId().longValue();
    private static final Long MOSCOW_AND_MOSCOW_REGION = RegionIDValues.MOSCOW_AND_MOSCOW_REGION.getId().longValue();
    private static final Long BELARUS = RegionIDValues.BELARUS.getId().longValue();
    private static final Long TURKEY = RegionIDValues.TURKEY.getId().longValue();
    private static final Long UKRAINE = RegionIDValues.UKRAINE.getId().longValue();

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(clientLogin);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter()
    public CampaignType campaignType;

    @Parameterized.Parameter(1)
    public AdGroupType adGroupType;

    @Parameterized.Parameter(2)
    public BannerType adType;

    private Long adGroupId;

    @Parameterized.Parameters(name = "campaign type [{0}], group type [{1}], banner type [{2}]")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {CampaignType.TEXT, AdGroupType.BASE, BannerType.TEXT},
                {CampaignType.TEXT, AdGroupType.BASE, BannerType.IMAGE_AD},
                {CampaignType.MOBILE_CONTENT, AdGroupType.MOBILE_CONTENT, BannerType.MOBILE_CONTENT},
                {CampaignType.MOBILE_CONTENT, AdGroupType.MOBILE_CONTENT, BannerType.IMAGE_AD},
                {CampaignType.DYNAMIC, AdGroupType.DYNAMIC, BannerType.DYNAMIC},
                {CampaignType.CPM_BANNER, AdGroupType.CPM_BANNER, BannerType.CPM_BANNER},
        });
    }

    @Before
    @Step("Подготовка данных для всех тестов")
    public void preparePerTestData() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(clientLogin);

        Long campaignId = api.userSteps.campaignSteps().addDefaultCampaign(campaignType);
        if (campaignType == CampaignType.CPM_BANNER) {
            api.userSteps.campaignFakeSteps().setType(campaignId, CampaignsType.CPM_BANNER);
        }

        adGroupId = api.userSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId).withDefaultGroup(adGroupType)
                        .withRegionIds(TURKEY));

        createAdWithMinusRegions(adGroupId, new Long[]{RUSSIA});
        createAdWithMinusRegions(adGroupId, new Long[]{UKRAINE});
    }

    private void createAdWithMinusRegions(Long adGroupId, Long[] adMinusRegionIds) {
        Long adId;
        if (adType.equals(BannerType.IMAGE_AD)) {
            String imageHash = api.userSteps.imagesStepsV5().addDefaultImageAdAdImage();
            adId = api.userSteps.adsSteps()
                    .addAdWithAdImageHash(adGroupId, adGroupType, BannerType.IMAGE_AD, imageHash);
        } else if (adType.equals(BannerType.CPM_BANNER)) {
            Long creativeId = api.userSteps.getDirectJooqDbSteps().useShardForLogin(clientLogin).perfCreativesSteps()
                    .saveDefaultCanvasCreativesForClient(Long.parseLong(User.get(clientLogin).getClientID()));
            adId = api.userSteps.adsSteps().addDefaultCpmBannerAdBuilderAd(adGroupId, creativeId);
        } else {
            adId = api.userSteps.adsSteps().addDefaultAd(adGroupId, adType);
        }

        api.userSteps.getDirectJooqDbSteps().useShardForLogin(clientLogin).bannersSteps()
                .saveBannersMinusGeo(adId, BannersMinusGeoType.current, stringifyGeoRegions(adMinusRegionIds));
    }

    private String stringifyGeoRegions(Long... ids) {
        return String.join(",", Arrays.stream(ids).map(Object::toString).collect(Collectors.toList()));
    }

    @Test
    @Description("У объявлений группы нет минус-регионов - RestrictedRegionIds пуст")
    public void warningAboutRestrictedRegions() {
        api.userSteps.adGroupsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAdGroups(
                        new AdGroupUpdateItemMap().withId(adGroupId).withRegionIds(MOSCOW_AND_MOSCOW_REGION)),
                ExpectedResult
                        .warnings(new Notification(10172, Api5ErrorDetails.ADS_WONT_BE_SHOWN_DUE_TO_LEGAL_RESTRICTIONS))
        );
    }

    @Test
    @Description("Минус-регион группы не включается в RestrictedRegionIds")
    public void noWarningAboutRestrictedRegions() {
        api.userSteps.adGroupsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap()
                        .withAdGroups(new AdGroupUpdateItemMap().withId(adGroupId).withRegionIds(BELARUS)),
                ExpectedResult.success(adGroupId)
        );
    }
}
