package ru.yandex.autotests.direct.api.adimages.delete;

import java.util.List;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adimages.AdImagesFeatures;
import ru.yandex.autotests.direct.api.adimages.AdImagesLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.Image;
import ru.yandex.autotests.directapi.model.ValidImageData;
import ru.yandex.autotests.directapi.model.api5.adimages.AdImageAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adimages.AdImageExpectedResult;
import ru.yandex.autotests.directapi.model.api5.adimages.AdImageHashesCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.adimages.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

/**
 * Created by lightelfik on 11.05.16.
 * https://st.yandex-team.ru/TESTIRT-9152
 */
@Aqua.Test
@Features(AdImagesFeatures.DELETE)
@Description("Проверка возможности удаления изображений")
@Issue("https://st.yandex-team.ru/DIRECT-47646")
public class AdImagesCheckAbilityToDelete {
    private static final String CLIENT = AdImagesLogins.CLIENT_DELETE_3;
    private static final String OTHER_CLIENT = AdImagesLogins.CLIENT_DELETE_1;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    public static void initTest() {
        api.as(CLIENT).userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        api.as(OTHER_CLIENT).userSteps.clientFakeSteps().fakeClearClientSpentUnits(OTHER_CLIENT);
    }

    public void associateImage(String login, String hash) {
        Long campaignId = api.as(login).userSteps.campaignSteps().addDefaultTextCampaign();
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        api.userSteps.adsSteps().addAd(
                new AdAddItemMap()
                        .withTextAd(
                                new TextAdAddMap()
                                        .defaultTextAd()
                                        .withAdImageHash(hash))
                        .withAdGroupId(adGroupId)
        );
    }

    public List<String> getImages(String login, int imageAmount) {
        List<String> hashes = api.as(login).userSteps.imagesStepsV5().addImagesForUserIfNotExists(login, imageAmount);
        return hashes;
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2333")
    public void oneNotAssociatedImage() {
        List<String> hashes = getImages(CLIENT, 1);
        String notAssociatedHash = hashes.get(0);

        api.as(CLIENT).userSteps.imagesStepsV5().shouldGetResultOnDelete(
                new DeleteRequestMap().withSelectionCriteria(
                        new AdImageHashesCriteriaMap()
                                .withAdImageHashes(notAssociatedHash)
                ),
                AdImageExpectedResult.success()
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2334")
    public void oneAssociatedImage() {
        //DIRECT-55566
        List<String> hashes = getImages(CLIENT, 1);
        String associatedHash = hashes.get(0);
        associateImage(CLIENT, associatedHash);

        api.as(CLIENT).userSteps.imagesStepsV5().shouldGetResultOnDelete(
                new DeleteRequestMap().withSelectionCriteria(
                        new AdImageHashesCriteriaMap()
                                .withAdImageHashes(associatedHash)
                ),
                AdImageExpectedResult.errors(
                        new Notification(8301,
                                Api5ErrorDetails.UNABLE_TO_DELETE_ASSOCIATED_IMAGE
                        )
                )
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2332")
    public void oneNotOwnedImage() {
        List<String> hashes = getImages(OTHER_CLIENT, 1);
        String notOwnedHash = hashes.get(0);

        api.as(CLIENT).userSteps.imagesStepsV5().shouldGetResultOnDelete(
                new DeleteRequestMap().withSelectionCriteria(
                        new AdImageHashesCriteriaMap()
                                .withAdImageHashes(notOwnedHash)
                ),
                AdImageExpectedResult.errors(
                        new Notification(8800,
                                Api5ErrorDetails.IMAGE_NOT_FOUND
                        )
                )
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2335")
    public void twoNotAssociatedImage() {
        List<String> hashes = getImages(CLIENT, 2);

        api.as(CLIENT).userSteps.imagesStepsV5().shouldGetResultOnDelete(
                new DeleteRequestMap().withSelectionCriteria(
                        new AdImageHashesCriteriaMap()
                                .withAdImageHashes(hashes.get(0), hashes.get(1))
                ),
                AdImageExpectedResult.success(), AdImageExpectedResult.success()
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2336")
    public void notAssociatedAndAssociatedImages() {
        List<String> hashes = getImages(CLIENT, 2);
        String associatedHash = hashes.get(0);
        String notAssociatedHash = hashes.get(1);
        associateImage(CLIENT, associatedHash);

        api.as(CLIENT).userSteps.imagesStepsV5().shouldGetResultOnDelete(
                new DeleteRequestMap().withSelectionCriteria(
                        new AdImageHashesCriteriaMap()
                                .withAdImageHashes(notAssociatedHash, associatedHash)
                ),
                AdImageExpectedResult.success(),
                AdImageExpectedResult.errors(
                        new Notification(8301,
                                Api5ErrorDetails.UNABLE_TO_DELETE_ASSOCIATED_IMAGE
                        ))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2337")
    public void notOwnedAndNotAssociatedImages() {
        Image notOwnedImage = ValidImageData.getValidWideImage(2);
        List<String> notOwnedhashes = api.as(OTHER_CLIENT).userSteps.imagesStepsV5().adImagesAdd(
                new AdImageAddItemMap()
                        .withImageData(notOwnedImage.getUrl())
                        .withName(notOwnedImage.getName())
        );
        String notOwnedHash = notOwnedhashes.get(0);

        Image image = ValidImageData.getValidWideImage(3);
        List<String> hashes = api.as(CLIENT).userSteps.imagesStepsV5().adImagesAdd(
                new AdImageAddItemMap()
                        .withImageData(image.getUrl())
                        .withName(image.getName())
        );
        String notAssociatedHash = hashes.get(0);

        api.as(CLIENT).userSteps.imagesStepsV5().shouldGetResultOnDelete(
                new DeleteRequestMap().withSelectionCriteria(
                        new AdImageHashesCriteriaMap()
                                .withAdImageHashes(notOwnedHash, notAssociatedHash)
                ),
                AdImageExpectedResult.errors(
                        new Notification(8800,
                                Api5ErrorDetails.IMAGE_NOT_FOUND
                        )),
                AdImageExpectedResult.success()
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2338")
    public void associatedAndNotOwnedImages() {
        Image image = ValidImageData.getValidRegularImage(11);
        List<String> hashes = api.as(OTHER_CLIENT).userSteps.imagesStepsV5().adImagesAdd(
                new AdImageAddItemMap()
                        .withImageData(image.getUrl())
                        .withName(image.getName())
        );
        String notOwnedHash = hashes.get(0);
        hashes = getImages(CLIENT, 1);
        String associatedHash = hashes.get(0);
        associateImage(CLIENT, associatedHash);

        api.as(CLIENT).userSteps.imagesStepsV5().shouldGetResultOnDelete(
                new DeleteRequestMap().withSelectionCriteria(
                        new AdImageHashesCriteriaMap()
                                .withAdImageHashes(associatedHash, notOwnedHash)
                ),
                AdImageExpectedResult.errors(
                        new Notification(8301,
                                Api5ErrorDetails.UNABLE_TO_DELETE_ASSOCIATED_IMAGE
                        )),
                AdImageExpectedResult.errors(
                        new Notification(8800,
                                Api5ErrorDetails.IMAGE_NOT_FOUND
                        ))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2339")
    public void oneImageTwice() {
        List<String> hashes = getImages(CLIENT, 1);
        String hash = hashes.get(0);

        api.as(CLIENT).userSteps.imagesStepsV5().shouldGetResultOnDelete(
                new DeleteRequestMap().withSelectionCriteria(
                        new AdImageHashesCriteriaMap()
                                .withAdImageHashes(hash, hash)),
                AdImageExpectedResult.errors(
                        new Notification(9800,
                                Api5ErrorDetails.IMAGE_HASH_IS_PRESENTED_MORE_THAN_ONCE_IN_REQUEST
                        )),
                AdImageExpectedResult.errors(
                        new Notification(9800,
                                Api5ErrorDetails.IMAGE_HASH_IS_PRESENTED_MORE_THAN_ONCE_IN_REQUEST
                        ))
        );
    }

}


