package ru.yandex.autotests.direct.api.ads;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;

import static org.apache.commons.lang3.RandomStringUtils.randomAlphabetic;
import static ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult.bothJavaAndPerl;

/**
 * Created by gerdler on 07.07.16.
 */

public class AdsHrefValidationHelper {

    private AdsHrefValidationHelper() {
    }

    private static final int MIN_DOMAIN_LENGTH = TextAdAddMap.MIN_DOMAIN_LENGTH;
    private static final int MAX_DOMAIN_LENGTH = TextAdAddMap.MAX_DOMAIN_LENGTH;
    private static final int MIN_TOP_DOMAIN_LENGTH = TextAdAddMap.MIN_TOP_DOMAIN_LENGTH;
    private static final int MAX_TOP_DOMAIN_LENGTH = TextAdAddMap.MAX_TOP_DOMAIN_LENGTH;
    private static final int MAX_HREF_LENGTH = TextAdAddMap.MAX_HREF_LENGTH;

    private static final String DEFAULT_DESCRIPTION_PREFIX = "Создать баннер со ссылкой";

    public static Collection<Object[]> getHrefDataForPositiveParameterized(String descriptionPrefix) {
        String httpHref = "http://yandex.ru";
        String httpsHref = "https://ya.ru";
        String minDomainLengthHref = "http://" + randomAlphabetic(MIN_DOMAIN_LENGTH) + ".ru";
        String maxDomainLengthHref = "http://" + randomAlphabetic(MAX_DOMAIN_LENGTH) +
                "." + randomAlphabetic(MAX_DOMAIN_LENGTH) + ".ru";
        String minTopDomainLengthHref = "http://test." + randomAlphabetic(MIN_TOP_DOMAIN_LENGTH);
        String maxTopDomainLengthHref = "http://tester." + randomAlphabetic(MAX_TOP_DOMAIN_LENGTH);
        String russianHref = "http://подарки.рф";
        String ukrainianHref = "http://сувеніри.укр";
        String numericHref = "http://12345.b45";
        String dashedHref = "http://hello-world.ru";
        String maxLengthHref = "http://" + generateValidHrefWithoutProtocolOfLength(MAX_HREF_LENGTH - 7);
        String hrefWithVariable = "http://yandex.ru/campaign={campaign_id}";

        return new ArrayList<>(Arrays.asList(new Object[][]{
                {descriptionPrefix + ", протокол которой http", httpHref},
                {descriptionPrefix + ", протокол которой https", httpsHref},
                {descriptionPrefix + " с внутренним доменом длиной " + MIN_DOMAIN_LENGTH,
                        minDomainLengthHref},
                {descriptionPrefix + " с внутренними доменами длиной " + MAX_DOMAIN_LENGTH,
                        maxDomainLengthHref},
                {descriptionPrefix + " с доменом верхнего уровня длиной " + MIN_TOP_DOMAIN_LENGTH,
                        minTopDomainLengthHref},
                {descriptionPrefix + " с доменом верхнего уровня длиной " + MAX_TOP_DOMAIN_LENGTH,
                        maxTopDomainLengthHref},
                {descriptionPrefix + ", состоящей из русских букв", russianHref},
                {descriptionPrefix + ", состоящей из украинских букв", ukrainianHref},
                {descriptionPrefix + ", содержащей цифры", numericHref},
                {descriptionPrefix + ", содержащей дефисы", dashedHref},
                {descriptionPrefix + ", содержащей подстановочную переменную", hrefWithVariable},
                {descriptionPrefix + ", длиной " + MAX_HREF_LENGTH, maxLengthHref}
        }));
    }

    public static List<Object[]> getHrefDataForNegativeParameterized(String descriptionPrefix) {
        String noProtocolHref = "example.com";
        String wrongProtocolHref = "ssh://example.com";
        String onlyNumericHref = "http://1234.56";
        String hrefWithSpecificSymbol = "http://hello!world.ru";
        String lessThanMinDomainLengthHref = "http://" + randomAlphabetic(MIN_DOMAIN_LENGTH - 1) + ".ru";
        String greaterThanMaxDomainLengthHref = "http://" + randomAlphabetic(MAX_DOMAIN_LENGTH + 1) +
                "." + randomAlphabetic(MIN_DOMAIN_LENGTH) + ".ru";
        String lessThanMinTopDomainLengthHref =
                "http://aaaaa." + randomAlphabetic(MIN_TOP_DOMAIN_LENGTH - 1);
        String greaterThanMaxTopDomainLengthHref =
                "http://tester." + randomAlphabetic(MAX_TOP_DOMAIN_LENGTH + 1);
        String greaterThanMaxLengthHref =
                "http://" + generateValidHrefWithoutProtocolOfLength(MAX_HREF_LENGTH - 6);
        String emptyHref = "";

        return new ArrayList<>(Arrays.asList((Object[][]) new Object[][]{
                {descriptionPrefix + " без протокола", noProtocolHref,
                        bothJavaAndPerl(5005, Api5ErrorDetails.INCORRECT_LINK, wrongProtocolHref)},

                {descriptionPrefix + ", протокол которой отличен от http или https", wrongProtocolHref,
                        bothJavaAndPerl(5005, Api5ErrorDetails.INCORRECT_LINK, wrongProtocolHref)},

                {descriptionPrefix + ", состоящей только из цифр", onlyNumericHref,
                        bothJavaAndPerl(5005, Api5ErrorDetails.INCORRECT_LINK, onlyNumericHref)},

                {descriptionPrefix + ", содержащей специфичный символ", hrefWithSpecificSymbol,
                        bothJavaAndPerl(5005, Api5ErrorDetails.INCORRECT_LINK, hrefWithSpecificSymbol)},

                {descriptionPrefix + " с внутренним доменом длиной меньшей, чем " +
                        MIN_DOMAIN_LENGTH, lessThanMinDomainLengthHref,
                        bothJavaAndPerl(5005,
                        Api5ErrorDetails.INCORRECT_LINK, lessThanMinDomainLengthHref)},

                {descriptionPrefix + " с внутренними доменами длиной большей, чем " +
                        MAX_DOMAIN_LENGTH, greaterThanMaxDomainLengthHref,
                        bothJavaAndPerl(5005,
                        Api5ErrorDetails.INCORRECT_LINK, greaterThanMaxDomainLengthHref)},

                {descriptionPrefix + " с доменом верхнего уровня длиной меньшей, чем" +
                        TextAdAddMap.MIN_TOP_DOMAIN_LENGTH, lessThanMinTopDomainLengthHref,
                        bothJavaAndPerl(5005,
                        Api5ErrorDetails.INCORRECT_LINK, lessThanMinTopDomainLengthHref)},

                {descriptionPrefix + " с доменом верхнего уровня длиной большей, чем" +
                        TextAdAddMap.MAX_TOP_DOMAIN_LENGTH, greaterThanMaxTopDomainLengthHref,
                        bothJavaAndPerl(5005,
                        Api5ErrorDetails.INCORRECT_LINK, greaterThanMaxTopDomainLengthHref)},

                {descriptionPrefix + ", длиной большей, чем " + TextAdAddMap.MAX_HREF_LENGTH,
                        greaterThanMaxLengthHref,
                        JavaOrPerlExpectedResult
                                .java(ExpectedResult.errors(new Notification(5005,
                                        Api5ErrorDetailsJava.LINK_HAS_EXCEEDED_CHARACTER_LIMIT, TextAdAddMap.MAX_HREF_LENGTH)))
                                .perl(ExpectedResult.errors(new Notification(5005,
                                        Api5ErrorDetails.LINK_HAS_EXCEEDED_CHARACTER_LIMIT, TextAdAddMap.MAX_HREF_LENGTH)))
                },
                {descriptionPrefix + ", равной пустой строке", wrongProtocolHref,
                        bothJavaAndPerl(5005, Api5ErrorDetails.INCORRECT_LINK, emptyHref)}
        }));
    }

    public static Collection<Object[]> getHrefDataForNegativeParameterized() {
        return getHrefDataForNegativeParameterized(DEFAULT_DESCRIPTION_PREFIX);
    }

    public static Collection<Object[]> getHrefDataForPositiveParameterized() {
        return getHrefDataForPositiveParameterized(DEFAULT_DESCRIPTION_PREFIX);
    }

    private static String generateValidHrefWithoutProtocolOfLength(int length) {
        StringBuilder stringBuilder = new StringBuilder(randomAlphabetic(length));
        int domainLength = Math.min(
                (TextAdAddMap.MAX_DOMAIN_LENGTH - 1) / 2, (TextAdAddMap.MAX_TOP_DOMAIN_LENGTH - 1) / 2);
        for (int i = domainLength; i < length - domainLength; i += domainLength) {
            stringBuilder.setCharAt(i, '.');
        }
        String result = stringBuilder.toString();
        return result;
    }

}
