package ru.yandex.autotests.direct.api.ads.add;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by pavryabov on 14.04.15.
 * https://st.yandex-team.ru/TESTIRT-5093
 */
@Aqua.Test
@Features(AdsFeatures.ADD)
@Description("Проверка правильности списания баллов")
@Issue("https://st.yandex-team.ru/DIRECT-34558")
public class AdsAddUnitsTest {
    private static String client = AdsLogins.ADD_UNITS;

    private static final int CALL_COST = 20;
    private static final int AD_COST = 20;
    private static final int DEFAULT_ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR = 50;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().version(104).as(client);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    static Long campaignId;
    static Long adGroupId;

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
    }

    @Test
    public void addOneAd() {
        //DIRECT-41089
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.adsSteps().adsAdd(new AddRequestMap().withAds(
                new AdAddItemMap()
                        .withTextAd(new TextAdAddMap().defaultTextAd())
                        .withAdGroupId(adGroupId)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + AD_COST));
    }

    @Test
    public void addTwoAds() {
        //DIRECT-41089
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.adsSteps().adsAdd(new AddRequestMap().withAds(
                new AdAddItemMap()
                        .withTextAd(new TextAdAddMap().defaultTextAd())
                        .withAdGroupId(adGroupId),
                new AdAddItemMap()
                        .withTextAd(new TextAdAddMap().defaultTextAd())
                        .withAdGroupId(adGroupId)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + 2 * AD_COST));
    }

    @Test
    public void addOneAdWithError() {
        //DIRECT-37674
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.adsSteps().shouldGetResultOnAdd(new AddRequestMap().withAds(
                        new AdAddItemMap()
                                .withTextAd(new TextAdAddMap().defaultTextAd())
                                .withAdGroupId(adGroupId + 100000)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.GROUP_NOT_FOUND))
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + DEFAULT_ERROR_COST));
    }

    @Test
    public void addTwoAdsWithOneError() {
        //DIRECT-37674
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.adsSteps().shouldGetResultOnAdd(new AddRequestMap().withAds(
                        new AdAddItemMap()
                                .withTextAd(new TextAdAddMap().defaultTextAd())
                                .withAdGroupId(adGroupId),
                        new AdAddItemMap()
                                .withTextAd(new TextAdAddMap().defaultTextAd())
                                .withAdGroupId(adGroupId + 100000)),
                ExpectedResult.success(),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.GROUP_NOT_FOUND))
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + DEFAULT_ERROR_COST + AD_COST));
    }

    @Test
    public void addTwoAdsWithTwoErrors() {
        //DIRECT-37674
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.adsSteps().shouldGetResultOnAdd(new AddRequestMap().withAds(
                        new AdAddItemMap()
                                .withTextAd(new TextAdAddMap().defaultTextAd())
                                .withAdGroupId(adGroupId + 100000),
                        new AdAddItemMap()
                                .withTextAd(new TextAdAddMap().defaultTextAd())
                                .withAdGroupId(adGroupId + 100000)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.GROUP_NOT_FOUND)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.GROUP_NOT_FOUND))
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + 2 * DEFAULT_ERROR_COST));
    }

    @Test
    public void addWithInvalidRequest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.adsSteps().expectErrorOnAdsAdd(new AddRequestMap()
                        .withAds(new AdAddItemMap()
                                .withTextAd(new TextAdAddMap()
                                        .withTitle(TextAdAddMap.DEFAULT_TITLE)
                                        .withHref(TextAdAddMap.DEFAULT_HREF)
                                        .withText(TextAdAddMap.DEFAULT_TEXT))
                                .withAdGroupId(adGroupId)),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        capitalize(TextAdAddMap.MOBILE)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(COMMON_REQUEST_ERROR));
    }

    @Test
    public void callAdsAddWithZeroUnits() {
        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(client, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assumeThat("у клиента не осталось баллов", units, equalTo(0));
        api.userSteps.adsSteps().expectErrorOnAdsAdd(new AddRequestMap().withAds(
                        new AdAddItemMap()
                                .withTextAd(new TextAdAddMap().defaultTextAd())
                                .withAdGroupId(adGroupId)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION)
        );
    }
}
