package ru.yandex.autotests.direct.api.ads.add;

import java.util.Arrays;
import java.util.Collection;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.utils.model.ShardNumbers;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.RandomStringUtils.randomAlphabetic;

/**
 * Created by pavryabov on 10.03.16.
 * https://st.yandex-team.ru/TESTIRT-8642
 */
@Aqua.Test
@Features(AdsFeatures.ADD)
@Description("Проверка прав доступа агентства к AdExtensionIds")
@Issue("https://st.yandex-team.ru/DIRECT-50726")
@RunWith(Parameterized.class)
public class AgencyCheckRightsToAdExtensionIdsTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static String client = AdsLogins.CLIENT;
    private static String agency = AdsLogins.AGENCY;
    private static String agencyRep = AdsLogins.AGENCY_REP;
    private static String agencyRepCl = AdsLogins.AGENCY_REP_CL;
    private static String subclient = AdsLogins.SUBCLIENT;
    private static String subclientElse = AdsLogins.SUBCLIENT_ELSE;

    static Long clientAdExtensionId;
    static Long subclientAdExtensionId;
    static Long subclientElseAdExtensionId;
    static Long adGroupId;

    @Parameterized.Parameter(0)
    public String loginForAuthorization;

    @Parameterized.Parameters(name = "loginForAuthorization = {0}")
    public static Collection testData() {
        Object[][] data = new Object[][]{
                {agency},
                {agencyRep},
                {agencyRepCl}
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(agency);
        api.userSteps.clientFakeSteps().reshardUserWithLock(subclient, ShardNumbers.DEFAULT_SHARD);
        api.userSteps.clientFakeSteps().reshardUserWithLock(subclientElse, ShardNumbers.DEFAULT_SHARD);
        api.as(client);
        clientAdExtensionId = api.userSteps.adExtensionsSteps().addCalloutWithText(randomAlphabetic(15));
        api.as(agency);
        subclientAdExtensionId = api.userSteps.adExtensionsSteps().addCalloutWithText(randomAlphabetic(15), subclient);
        subclientElseAdExtensionId = api.userSteps.adExtensionsSteps().addCalloutWithText(randomAlphabetic(15), subclientElse);
        Long subclientCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(subclient);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(subclientCampaignId, subclient);
    }

    @Test
    public void callAddWithSubclientAdExtensionIds() {
        api.as(loginForAuthorization).userSteps.adsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAds(
                        new AdAddItemMap()
                                .withTextAd(new TextAdAddMap().defaultTextAd().withAdExtensionIds(subclientAdExtensionId))
                                .withAdGroupId(adGroupId)
                ),
                subclient,
                ExpectedResult.success()
        );
    }

    @Test
    public void callAddWithSubclientElseAdExtensionIds() {
        api.as(loginForAuthorization).userSteps.adsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAds(
                        new AdAddItemMap()
                                .withTextAd(new TextAdAddMap().defaultTextAd().withAdExtensionIds(subclientElseAdExtensionId))
                                .withAdGroupId(adGroupId)
                ),
                subclient,
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AD_EXTENSION_NOT_FOUND, subclientElseAdExtensionId))
        );
    }

    @Test
    public void callAddWithAdExtensionIdsOfBothSubclients() {
        api.as(loginForAuthorization).userSteps.adsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAds(
                        new AdAddItemMap()
                                .withTextAd(new TextAdAddMap().defaultTextAd().withAdExtensionIds(subclientAdExtensionId))
                                .withAdGroupId(adGroupId),
                        new AdAddItemMap()
                                .withTextAd(new TextAdAddMap().defaultTextAd().withAdExtensionIds(subclientElseAdExtensionId))
                                .withAdGroupId(adGroupId)
                ),
                subclient,
                ExpectedResult.success(),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AD_EXTENSION_NOT_FOUND, subclientElseAdExtensionId))
        );
    }

    @Test
    public void callAddWithClientAdExtensionIds() {
        api.as(loginForAuthorization).userSteps.adsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAds(
                        new AdAddItemMap()
                                .withTextAd(new TextAdAddMap().defaultTextAd().withAdExtensionIds(clientAdExtensionId))
                                .withAdGroupId(adGroupId)
                ),
                subclient,
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AD_EXTENSION_NOT_FOUND, clientAdExtensionId))
        );
    }
}
