package ru.yandex.autotests.direct.api.ads.add;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.yandex.direct.api.v5.ads.AddResponse;
import one.util.streamex.EntryStream;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersBannerType;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.DynamicTextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppImageAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextImageAdAddMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.banners.BannerType;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Collections.singletonList;
import static org.apache.commons.lang.StringUtils.uncapitalize;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by pavryabov on 07.04.15.
 * https://st.yandex-team.ru/TESTIRT-5093
 */
@Aqua.Test
@Features(AdsFeatures.ADD)
@Description("Проверка правильности сохранения типа")
@Issue("https://st.yandex-team.ru/DIRECT-34558")
@Tag(TagDictionary.TRUNK)
public class SaveTypeTest {

    private static final String LOGIN = AdsLogins.CLIENT47;
    private static final Map<BannersBannerType, String> TYPE_DESCRIPTIONS = ImmutableMap.of(
            BannersBannerType.text, "текстовый",
            BannersBannerType.mobile_content, "РМП",
            BannersBannerType.dynamic, "динамический",
            BannersBannerType.image_ad, "графический"
    );

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    static Integer shard;

    static Long campaignId;
    static Long adGroupId;

    static Long mobileAppCampaignId;
    static Long mobileAppAdGroupId;

    static Long dynamicTextCampaignId;
    static Long dynamicTextAdGroupId;

    static String adImageHash;

    @BeforeClass
    public static void initTest() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);

        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);

        mobileAppCampaignId = api.userSteps.campaignSteps().addDefaultMobileAppCampaign();
        mobileAppAdGroupId = api.userSteps.adGroupsSteps().addDefaultMobileGroup(mobileAppCampaignId);

        dynamicTextCampaignId = api.userSteps.campaignSteps().addDefaultDynamicTextCampaign();
        dynamicTextAdGroupId = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(dynamicTextCampaignId);

        adImageHash = api.userSteps.imagesStepsV5().addImageAdAdImage(2);
    }

    @Test
    public void saveTextAd() {
        Long id = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        BannerFakeInfo bannerFakeInfo =
                api.userSteps.getDarkSideSteps().getBannersFakeSteps().getBannerParams(id);
        assertThat("баннер сохранился с правильным типом",
                bannerFakeInfo.getType(), equalTo(uncapitalize(BannerType.DESKTOP)));

    }

    @Test
    public void saveMobileAd() {
        Long id = api.userSteps.adsSteps().addDefaultMobileAd(adGroupId);
        BannerFakeInfo bannerFakeInfo =
                api.userSteps.getDarkSideSteps().getBannersFakeSteps().getBannerParams(id);
        assertThat("баннер сохранился с правильным типом",
                bannerFakeInfo.getType(), equalTo(uncapitalize(BannerType.MOBILE)));

    }

    @Test
    public void saveMobileAppAd() {
        Long adId = api.userSteps.adsSteps().addDefaultMobileAppAd(mobileAppAdGroupId);
        checkBannerTypesInDb(singletonList(adId), singletonList(BannersBannerType.mobile_content));
    }

    @Test
    public void addTwoMobileAppAds() {
        List<Long> adIds = api.userSteps.adsSteps().addDefaultMobileAppAds(mobileAppAdGroupId, 2);
        assumeThat("было создано два объявления", adIds, hasSize(2));
        checkBannerTypesInDb(adIds, ImmutableList.of(BannersBannerType.mobile_content,
                BannersBannerType.mobile_content));
    }

    @Test
    public void addTwoMobileAppAdsTwoAdGroupsTest() {
        Long anotherMobileAppAdGroupId = api.userSteps.adGroupsSteps().addDefaultMobileGroup(mobileAppCampaignId);

        api.userSteps.adsSteps().shouldGetResultOnAdd(new AddRequestMap().withAds(
                new AdAddItemMap()
                        .withAdGroupId(mobileAppAdGroupId)
                        .withMobileAppAd(new MobileAppAdAddMap()
                                .defaultMobileAppAd()),
                new AdAddItemMap()
                        .withAdGroupId(anotherMobileAppAdGroupId)
                        .withMobileAppAd(new MobileAppAdAddMap()
                                .defaultMobileAppAd())),
                ExpectedResult.success(), ExpectedResult.success());
    }

    @Test
    public void saveTypesAfterTextAndMobileAdsInOneRequest() {
        AddResponse addResponse = api.userSteps.adsSteps().adsAdd(new AddRequestMap()
                .withAds(new AdAddItemMap()
                                .withTextAd(new TextAdAddMap().defaultTextAd())
                                .withAdGroupId(adGroupId),
                        new AdAddItemMap()
                                .withTextAd(new TextAdAddMap().defaultMobileAd())
                                .withAdGroupId(adGroupId)));
        List<Long> ids =
                addResponse.getAddResults().stream()
                        .map(actionResult -> actionResult.getId())
                        .collect(Collectors.toList());
        assumeThat("получено два id", ids, hasSize(2));
        BannerFakeInfo bannerFakeInfo =
                api.userSteps.getDarkSideSteps().getBannersFakeSteps().getBannerParams(ids.get(0));
        assertThat("текстовый баннер сохранился с правильным типом",
                bannerFakeInfo.getType(), equalTo(uncapitalize(BannerType.DESKTOP)));
        bannerFakeInfo =
                api.userSteps.getDarkSideSteps().getBannersFakeSteps().getBannerParams(ids.get(1));
        assertThat("мобильный баннер сохранился с правильным типом",
                bannerFakeInfo.getType(), equalTo(uncapitalize(BannerType.MOBILE)));

    }

    @Test
    public void saveDynamicTextAd() {
        Long id = api.userSteps.adsSteps().addDefaultDynamicTextAd(dynamicTextAdGroupId);
        checkBannerTypesInDb(singletonList(id), singletonList(BannersBannerType.dynamic));
    }

    @Test
    public void saveAdsOfAllTypes() {
        AddResponse addResponse = api.userSteps.adsSteps().adsAdd(new AddRequestMap()
                .withAds(new AdAddItemMap()
                                .withTextAd(new TextAdAddMap().defaultTextAd())
                                .withAdGroupId(adGroupId),
                        new AdAddItemMap()
                                .withMobileAppAd(new MobileAppAdAddMap().defaultMobileAppAd())
                                .withAdGroupId(mobileAppAdGroupId),
                        new AdAddItemMap()
                                .withDynamicTextAd(new DynamicTextAdAddMap().defaultDynamicTextAd())
                                .withAdGroupId(dynamicTextAdGroupId),
                        new AdAddItemMap()
                                .withTextImageAd(new TextImageAdAddMap().defaultTextImageAd(adImageHash))
                                .withAdGroupId(adGroupId),
                        new AdAddItemMap()
                                .withMobileAppImageAd(new MobileAppImageAdAddMap().defaultMobileAppImageAd(adImageHash))
                                .withAdGroupId(mobileAppAdGroupId)));
        List<Long> ids =
                addResponse.getAddResults().stream()
                        .map(actionResult -> actionResult.getId())
                        .collect(Collectors.toList());
        assumeThat("получено пять id", ids, hasSize(5));
        checkBannerTypesInDb(ids, ImmutableList.of(BannersBannerType.text, BannersBannerType.mobile_content,
                BannersBannerType.dynamic, BannersBannerType.image_ad, BannersBannerType.image_ad));
    }

    private void checkBannerTypesInDb(List<Long> adIds, List<BannersBannerType> expectedTypesList) {
        EntryStream.zip(adIds, expectedTypesList).forKeyValue((id, expectedType) -> {
            BannersBannerType actualType = api.userSteps.getDirectJooqDbSteps()
                    .useShardForLogin(LOGIN)
                    .bannersSteps()
                    .getBanner(id)
                    .getBannerType();
            String assertMessage = TYPE_DESCRIPTIONS.get(expectedType) + " баннер сохранился с правильным типом " +
                    "баннера";
            assertThat(assertMessage, actualType, is(expectedType));
        });
    }
}
