package ru.yandex.autotests.direct.api.ads.add.cpmbanneradbuilderad;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.function.Supplier;

import javax.util.streamex.StreamEx;

import com.google.common.collect.ImmutableList;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListRuleOperatorEnum;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListTypeEnum;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang3.RandomUtils;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.ClientPixelProvidersCriterionType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.PixelProvider;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.CpmBannerAdBuilderAdAddMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListAddItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListRuleArgumentItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListRuleItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Collections.singletonList;
import static ru.yandex.autotests.directapi.model.AdfoxDealPlacement.getPlacementJsonsFromPageIds;
import static ru.yandex.autotests.directapi.model.api5.general.ExpectedResult.success;

/**
 * https://st.yandex-team.ru/DIRECT-84893
 */
@Aqua.Test
@Features(AdsFeatures.ADD)
@Description("Проверка валидации пикселей аудита")
@Issue("https://st.yandex-team.ru/DIRECT-84893")
@RunWith(Parameterized.class)
public class AddCpmAdPixelValidationTest {

    private static final String clientLogin = AdsLogins.SUBCLIENT;
    private static final String agencyLogin = AdsLogins.AGENCY;
    private static final Long PUBLIC_GOAL_ID = 2499000002L;
    private static final Long PRIVATE_GOAL_ID = 2499000010L;
    private static final Long MIN_TEST_DEAL_ID = 2000000L;
    private static final Long MAX_TEST_DEAL_ID = 10000000L;
    private static final Long BIG_PLACEMENT_PAGE_ID = 63545390L;
    private static final Long CPM_VIDEO_LAYOUT_ID = 6L;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(agencyLogin);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long rsyaCampaignId;
    private static Long foreignInventoryCampaignId;

    @Parameterized.Parameter()
    public String description;

    @Parameterized.Parameter(1)
    public List<TestData> testDataList;

    private static Long cpmBannerCreativeId;
    private static Long cpmVideoCreativeId;


    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
               {"RSYA campaign with adfox tracking url", singletonList(getTestDataForRsyaAdfoxPositiveTest())},
                {"CPM banner campaign with admetrica tracking url",
                        singletonList(getTestDataCpmBannerCampaignAdmetricaPixelPositiveTest())},
                {"CPM banner campaign with new weborama tracking url",
                        singletonList(getTestDataCpmBannerCampaignWeboramaRuPixelPositiveTest())},
                {"CPM banner campaign with amc admetrica tracking url",
                        singletonList(getTestDataCpmBannerCampaignAmcAdmetricaPixelPositiveTest())},
                {"Invalid pixel with linebreak in url test",
                        singletonList(getTestDataInvalidPixelTwoStringsInOnePixel())},
                {"Invalid pixel url test", singletonList(getTestDataInvalidPixelUrl())},
                {"RSYA campaign with adfox tracking url and " +
                        "public-segment deal with foreign inventory adfox, yndx_audience, pixels",
                        ImmutableList.of(getTestDataForRsyaAdfoxPositiveTest(),
                                getTestDataForMixedPixelPublicSegmentDealNegativeTest())
                },
                {"Private-segment deal with foreign inventory yndx_audience, adriver pixels positive test",
                        ImmutableList.of(getTestDataForMixedPixelPrivateSegmentDealPositiveTest())
                },
                {"Private-segment deal with foreign inventory yndx_audience, adriver pixels negative test",
                        ImmutableList.of(getTestDataForMixedPixelPrivateSegmentDealNegativeTest())
                },
                {"Keywords with foreign inventory yndx_audience, adriver pixels negative test",
                        ImmutableList.of(getTestDataForMixedPixelWithKeywordsNegativeTest())
                },
                {"Two not yaAudience pixels forbidden",
                        singletonList(getTestDataForPixelTwoNotYaAudienceTwoErrorsNegativeTest())},
                {"Two yaAudience pixels", singletonList(getTestDataForPixelYaAudienceLimitsNegativeTest())},
                {"Two not yaAudience pixels allowed",
                        singletonList(getTestDataForPixelTwoNotYaAudiencePixelsPositiveTest())},
                {"Two pixel providers repeated forbidden, two pixel providers repeated allowed",
                        singletonList(getTestDataForPixelTwoRepeatedNotAllowedTwoRepeatedAllowedTest())},
                {"Four not yaAudience pixels allowed",
                        singletonList(getTestDataForPixelFourNotYaAudiencePixelsPositiveTest())},
                {"Five not yaAudience pixels not allowed",
                        singletonList(getTestDataForPixelFiveNotYaAudiencePixelsNegativeTest())},
                {"Two not yaAudience pixels are allowed in cpmVideo",
                        singletonList(getTestDataForPixelTwoNotYaAudiencePixelsCpmVideoPositiveTest())},
                {"Five not yaAudience pixels not allowed in cpmVideo",
                        singletonList(getTestDataForPixelFiveNotYaAudiencePixelsCpmVideoNegativeTest())},
                {"Not yaAudience pixel allowed in cpmVideo",
                        singletonList(getTestDataForPixelNotYaAudiencePixelPositiveTest())}
        });
    }

    @BeforeClass
    @Step("Подготовка данных")
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(agencyLogin);
        rsyaCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(clientLogin);
        api.userSteps.campaignFakeSteps().setType(rsyaCampaignId,
                ru.yandex.autotests.directapi.darkside.model.CampaignsType.CPM_BANNER);
        foreignInventoryCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(clientLogin);
        api.userSteps.campaignFakeSteps().setType(foreignInventoryCampaignId,
                ru.yandex.autotests.directapi.darkside.model.CampaignsType.CPM_DEALS);
        addForeignPlacements(foreignInventoryCampaignId);

        cpmBannerCreativeId = api.userSteps.getDirectJooqDbSteps().useShardForLogin(clientLogin).perfCreativesSteps()
                .saveDefaultCanvasCreativesForClient(Long.parseLong(User.get(clientLogin).getClientID()));
        cpmVideoCreativeId = api.userSteps.getDirectJooqDbSteps().useShardForLogin(clientLogin).perfCreativesSteps()
                .saveDefaultVideoCreative(Long.parseLong(User.get(clientLogin).getClientID()), CPM_VIDEO_LAYOUT_ID);
    }

    private static void addForeignPlacements(Long cpmDealCampaignId) {
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(agencyLogin)
                .placementsSteps()
                .addPlacement(BIG_PLACEMENT_PAGE_ID, 0);
        Long randomDealId = RandomUtils.nextLong(MIN_TEST_DEAL_ID, MAX_TEST_DEAL_ID);
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(agencyLogin)
                .adfoxSteps()
                .addDeal(randomDealId, Long.parseLong(User.get(agencyLogin).getClientID()),
                        getPlacementJsonsFromPageIds(singletonList(BIG_PLACEMENT_PAGE_ID)));
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(agencyLogin)
                .dealsSteps()
                .addDeal(randomDealId, Long.parseLong(User.get(agencyLogin).getClientID()));
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(clientLogin)
                .campaignDealsSteps()
                .addCampaignDeal(randomDealId, cpmDealCampaignId, 0);
    }

    @Test
    @Description("Попытка добавить баннер с пикселями")
    public void testBannerWithPixelsAdd() {
        for (TestData testData : testDataList) {
            writeToClientPixelProviders(testData);
        }

        List<AdAddItemMap> adAddItemMaps = StreamEx.of(testDataList)
                .map(testData -> cpmAdAddItemMapByAdGroupAndPixel(testData.adGroupIds.get(),
                        StreamEx.of(testData.pixelProviders).map(PixelProvider::testUrl).toList(),
                        testData.isCpmBanner ? cpmBannerCreativeId : cpmVideoCreativeId))
                .toList();

        if (testDataList.size() == 1) {
            api.userSteps.adsSteps()
                    .shouldGetResultOnAdd(new AddRequestMap().withAds(adAddItemMaps), clientLogin,
                            testDataList.get(0).expectedResult);
        } else {
            api.userSteps.adsSteps()
                    .shouldGetResultOnAdd(new AddRequestMap().withAds(adAddItemMaps),
                            clientLogin,
                            testDataList.get(0).expectedResult,
                            testDataList.get(1).expectedResult);
        }

        deleteFromClientPixelProviders();
    }

    private AdAddItemMap cpmAdAddItemMapByAdGroupAndPixel(Long adGroupId, List<String> trackingPixels, Long creativeId) {
        return new AdAddItemMap()
                .withCpmBannerAdBuilderAd(new CpmBannerAdBuilderAdAddMap()
                        .withHref(CpmBannerAdBuilderAdAddMap.DEFAULT_HREF)
                        .withCreative(creativeId)
                        .withTrackingPixels(trackingPixels))
                .withAdGroupId(adGroupId);
    }

    private void writeToClientPixelProviders(TestData testData) {
        for (int i = 0; i < testData.pixelProviders.size(); ++i) {
            if (testData.shouldAddPermissionsToDb.get(i)) {
                api.userSteps.getDirectJooqDbSteps()
                        .useShardForLogin(clientLogin)
                        .clientPixelProvidersSteps()
                        .addClientPixelProvider(Long.parseLong(User.get(clientLogin).getClientID()),
                                testData.campaignType,
                                testData.inventoryTypeToDb,
                                testData.pixelProviders.get(i).pixelProviderFromDb());
            }
        }
    }

    private void deleteFromClientPixelProviders() {
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(clientLogin)
                .clientPixelProvidersSteps()
                .deleteClientProviders(Long.parseLong(User.get(clientLogin).getClientID()));
    }

    private static Long adGroupForRsyaTest() {
        return api.userSteps.adGroupsSteps().addDefaultCpmBannerUserProfileAdGroup(rsyaCampaignId, clientLogin);
    }

    private static Long adGroupForCpmVideoTest() {
        return api.userSteps.adGroupsSteps().addDefaultCpmVideoAdGroup(rsyaCampaignId, clientLogin);
    }

    private static Long adGroupWithKeywordsTest() {
        return api.userSteps.adGroupsSteps()
                .addDefaultCpmBannerKeywordsAdGroup(foreignInventoryCampaignId, clientLogin);
    }

    private static Long getForeignInventoryPrivateSegmentsAdGroup() {
        return getForeignInventoryAdGroupWithSingleGoal(PRIVATE_GOAL_ID);
    }

    private static Long getForeignInventoryPublicSegmentsAdGroup() {
        return getForeignInventoryAdGroupWithSingleGoal(PUBLIC_GOAL_ID);
    }

    private static Long getForeignInventoryAdGroupWithSingleGoal(Long goalId) {
        Long adGroupId =
                api.userSteps.adGroupsSteps()
                        .addDefaultCpmBannerUserProfileAdGroup(foreignInventoryCampaignId, clientLogin);
        RetargetingListAddItemMap retargetingListAddItemMap = new RetargetingListAddItemMap()
                .withType(RetargetingListTypeEnum.AUDIENCE)
                .withName("Name" + RandomStringUtils.randomAlphabetic(3))
                .withDescription("Description" + RandomStringUtils.randomAlphabetic(3))
                .withRules(new RetargetingListRuleItemMap()
                        .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                .withMembershipLifeSpan(RandomUtils.nextInt(1, 90))
                                .withExternalId(goalId))
                        .withOperator(RetargetingListRuleOperatorEnum.ANY));
        Long retargetingListId = api.userSteps.retargetingListsSteps().add(clientLogin,
                new ru.yandex.autotests.directapi.model.api5.retargetinglists.AddRequestMap()
                        .withRetargetingLists(retargetingListAddItemMap)).get(0);
        api.userSteps.audienceTargetsSteps().addWithRetargetingList(clientLogin, adGroupId, retargetingListId);
        return adGroupId;
    }

    private static TestData getTestDataForRsyaAdfoxPositiveTest() {
        return new TestData(singletonList(PixelProvider.ADFOX),
                AddCpmAdPixelValidationTest::adGroupForRsyaTest,
                success(),
                singletonList(false),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                true);
    }

    private static TestData getTestDataCpmBannerCampaignAdmetricaPixelPositiveTest() {
        return new TestData(singletonList(PixelProvider.ADFOX_ADMETRICA),
                AddCpmAdPixelValidationTest::adGroupForRsyaTest,
                success(),
                singletonList(false),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                true);
    }

    private static TestData getTestDataCpmBannerCampaignWeboramaRuPixelPositiveTest() {
        return new TestData(singletonList(PixelProvider.WEBORAMA),
                AddCpmAdPixelValidationTest::adGroupForRsyaTest,
                success(),
                singletonList(true),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                true);
    }

    private static TestData getTestDataForPixelNotYaAudiencePixelPositiveTest() {
        return new TestData(
                ImmutableList.of(PixelProvider.ADFOX),
                AddCpmAdPixelValidationTest::adGroupForCpmVideoTest,
                success(),
                ImmutableList.of(false),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                false);
    }

    private static TestData getTestDataForPixelTwoNotYaAudiencePixelsCpmVideoPositiveTest() {
        return new TestData(
                ImmutableList.of(PixelProvider.ADFOX, PixelProvider.ADRIVER),
                AddCpmAdPixelValidationTest::adGroupForCpmVideoTest,
                success(),
                ImmutableList.of(false, true),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                false);
    }

    private static TestData getTestDataForPixelFiveNotYaAudiencePixelsCpmVideoNegativeTest() {
        return new TestData(
                ImmutableList.of(PixelProvider.ADFOX, PixelProvider.ADFOX_ADMETRICA, PixelProvider.ADRIVER,
                        PixelProvider.TNS, PixelProvider.MC_ADMETRICA),
                AddCpmAdPixelValidationTest::adGroupForCpmVideoTest,
                ExpectedResult.errors(new Notification(6000,
                        Api5ErrorDetailsJava.MAX_FOREIGN_PIXELS_ON_BANNER)),
                ImmutableList.of(false, false, true, true, false),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                false);
    }

    private static TestData getTestDataCpmBannerCampaignAmcAdmetricaPixelPositiveTest() {
        return new TestData(singletonList(PixelProvider.MC_ADMETRICA),
                AddCpmAdPixelValidationTest::adGroupForRsyaTest,
                success(),
                singletonList(false),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                true);
    }

    private static TestData getTestDataInvalidPixelTwoStringsInOnePixel() {
        return new TestData(singletonList(PixelProvider.TWO_PIXEL_ONE_STRING),
                AddCpmAdPixelValidationTest::adGroupForRsyaTest,
                ExpectedResult.errors(new Notification(5004,
                        Api5ErrorDetailsJava.INVALID_PIXEL_FORMAT)),
                singletonList(false),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                true);
    }

    private static TestData getTestDataInvalidPixelUrl() {
        return new TestData(singletonList(PixelProvider.CHESTNY_AUDIT),
                AddCpmAdPixelValidationTest::adGroupForRsyaTest,
                ExpectedResult.errors(new Notification(5004,
                        Api5ErrorDetailsJava.INVALID_PIXEL_FORMAT)),
                singletonList(false),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                true);
    }

    private static TestData getTestDataForPixelTwoNotYaAudienceTwoErrorsNegativeTest() {
        return new TestData(
                ImmutableList.of(PixelProvider.TNS, PixelProvider.YANDEXAUDIENCE, PixelProvider.ADRIVER),
                AddCpmAdPixelValidationTest::adGroupForRsyaTest,
                ExpectedResult.errors(
                        new Notification(5004,
                                Api5ErrorDetailsJava.NO_RIGHTS_TO_PIXEL,
                                PixelProvider.TNS.testUrl()),
                        new Notification(5004,
                                Api5ErrorDetailsJava.NO_RIGHTS_TO_PIXEL,
                                PixelProvider.ADRIVER.testUrl())),
                ImmutableList.of(false, false, false),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                true);
    }

    private static TestData getTestDataForPixelTwoNotYaAudiencePixelsPositiveTest() {
        return new TestData(
                ImmutableList.of(PixelProvider.ADFOX, PixelProvider.YANDEXAUDIENCE, PixelProvider.DCM),
                AddCpmAdPixelValidationTest::adGroupForRsyaTest,
                success(),
                ImmutableList.of(false, false, true),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                true);
    }

    private static TestData getTestDataForPixelFourNotYaAudiencePixelsPositiveTest() {
        return new TestData(
                ImmutableList.of(PixelProvider.ADFOX, PixelProvider.YANDEXAUDIENCE, PixelProvider.ADRIVER,
                        PixelProvider.TNS, PixelProvider.DCM),
                AddCpmAdPixelValidationTest::adGroupForRsyaTest,
                success(),
                ImmutableList.of(false, false, true, true, true),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                true);
    }

    private static TestData getTestDataForPixelTwoRepeatedNotAllowedTwoRepeatedAllowedTest() {
        return new TestData(
                ImmutableList.of(PixelProvider.ADFOX, PixelProvider.MC_ADMETRICA,
                        PixelProvider.ADRIVER, PixelProvider.ADRIVER),
                AddCpmAdPixelValidationTest::adGroupForRsyaTest,
                ExpectedResult.errors(
                        new Notification(5004,
                                Api5ErrorDetailsJava.TOO_MANY_PIXELS_OF_SAME_PROVIDER,
                                PixelProvider.ADFOX.providerName()),
                        new Notification(5004,
                                Api5ErrorDetailsJava.NO_RIGHTS_TO_PIXEL,
                                PixelProvider.ADRIVER.testUrl()),
                        new Notification(5004,
                                Api5ErrorDetailsJava.NO_RIGHTS_TO_PIXEL,
                                PixelProvider.ADRIVER.testUrl())),
                ImmutableList.of(false, false, false, false, false),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                true);
    }

    private static TestData getTestDataForPixelFiveNotYaAudiencePixelsNegativeTest() {
        return new TestData(
                ImmutableList.of(PixelProvider.ADFOX, PixelProvider.ADFOX_ADMETRICA, PixelProvider.ADRIVER,
                        PixelProvider.TNS, PixelProvider.MC_ADMETRICA),
                AddCpmAdPixelValidationTest::adGroupForRsyaTest,
                ExpectedResult.errors(new Notification(6000,
                        Api5ErrorDetailsJava.MAX_FOREIGN_PIXELS_ON_BANNER)),
                ImmutableList.of(false, false, true, true, false),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                true);
    }

    private static TestData getTestDataForPixelYaAudienceLimitsNegativeTest() {
        return new TestData(
                ImmutableList.of(PixelProvider.ADFOX, PixelProvider.YANDEXAUDIENCE, PixelProvider.YANDEXAUDIENCE),
                AddCpmAdPixelValidationTest::adGroupForRsyaTest,
                ExpectedResult.errors(new Notification(6000,
                        Api5ErrorDetailsJava.MAX_YA_AUD_PIXELS_ON_BANNER)),
                ImmutableList.of(false, false, false),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.cpm_banner,
                true);
    }

    private static TestData getTestDataForMixedPixelPublicSegmentDealNegativeTest() {
        return new TestData(ImmutableList.of(PixelProvider.ADFOX, PixelProvider.YANDEXAUDIENCE),
                AddCpmAdPixelValidationTest::getForeignInventoryPublicSegmentsAdGroup,
                ExpectedResult.errors(new Notification(5004,
                        Api5ErrorDetailsJava.NO_RIGHTS_TO_AUDIENCE_PIXEL,
                        PixelProvider.YANDEXAUDIENCE.testUrl())),
                ImmutableList.of(false, false),
                ClientPixelProvidersCriterionType.public_,
                CampaignsType.cpm_deals,
                true);
    }

    private static TestData getTestDataForMixedPixelPrivateSegmentDealPositiveTest() {
        return new TestData(ImmutableList.of(PixelProvider.ADRIVER, PixelProvider.YANDEXAUDIENCE),
                AddCpmAdPixelValidationTest::getForeignInventoryPrivateSegmentsAdGroup,
                success(),
                ImmutableList.of(true, true),
                ClientPixelProvidersCriterionType.private_,
                CampaignsType.cpm_deals,
                true);
    }

    private static TestData getTestDataForMixedPixelPrivateSegmentDealNegativeTest() {
        return new TestData(ImmutableList.of(PixelProvider.ADRIVER, PixelProvider.YANDEXAUDIENCE),
                AddCpmAdPixelValidationTest::getForeignInventoryPrivateSegmentsAdGroup,
                ExpectedResult.errors(
                        new Notification(5004,
                                Api5ErrorDetailsJava.NO_RIGHTS_TO_PIXEL,
                                PixelProvider.ADRIVER.testUrl()),
                        new Notification(5004,
                                Api5ErrorDetailsJava.NO_RIGHTS_TO_AUDIENCE_PIXEL,
                                PixelProvider.YANDEXAUDIENCE.testUrl())),
                ImmutableList.of(false, false),
                ClientPixelProvidersCriterionType.private_,
                CampaignsType.cpm_deals,
                true);
    }

    private static TestData getTestDataForMixedPixelWithKeywordsNegativeTest() {
        return new TestData(ImmutableList.of(PixelProvider.ADRIVER, PixelProvider.YANDEXAUDIENCE),
                AddCpmAdPixelValidationTest::adGroupWithKeywordsTest,
                ExpectedResult.errors(
                        new Notification(5004,
                                Api5ErrorDetailsJava.NO_RIGHTS_TO_PIXEL,
                                PixelProvider.ADRIVER.testUrl()),
                        new Notification(5004,
                                Api5ErrorDetailsJava.NO_RIGHTS_TO_AUDIENCE_PIXEL,
                                PixelProvider.YANDEXAUDIENCE.testUrl())),
                ImmutableList.of(false, false),
                ClientPixelProvidersCriterionType.private_,
                CampaignsType.cpm_deals,
                true);
    }

    private static class TestData {
        public List<PixelProvider> pixelProviders;
        public Supplier<Long> adGroupIds;
        public ExpectedResult expectedResult;
        public List<Boolean> shouldAddPermissionsToDb;
        public ClientPixelProvidersCriterionType inventoryTypeToDb;
        public CampaignsType campaignType;
        public boolean isCpmBanner;


        public TestData(
                List<PixelProvider> pixelProviders, Supplier<Long> adGroupIds,
                ExpectedResult expectedResult, List<Boolean> shouldAddPermissionsToDb,
                ClientPixelProvidersCriterionType inventoryTypeToDb,
                CampaignsType campaignType, boolean isCpmBanner)
        {
            this.pixelProviders = pixelProviders;
            this.adGroupIds = adGroupIds;
            this.expectedResult = expectedResult;
            this.shouldAddPermissionsToDb = shouldAddPermissionsToDb;
            this.inventoryTypeToDb = inventoryTypeToDb;
            this.campaignType = campaignType;
            this.isCpmBanner = isCpmBanner;
        }
    }

}
