package ru.yandex.autotests.direct.api.ads.add.mobileappad;

import com.yandex.direct.api.v5.ads.MobileAppFeatureEnum;
import com.yandex.direct.api.v5.general.MobileAppAdActionEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersMobileContentPrimaryAction;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PhrasesAdgroupType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersMobileContentRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.PhrasesRecord;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.enums.ReflectedAttrs;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppAdFeatureItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasProperty;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by semkagtn on 26.08.15.
 * https://st.yandex-team.ru/TESTIRT-6853
 */
@Aqua.Test
@Features(AdsFeatures.ADD)
@Description("Проверка правильности сохранения значений в базе данных " +
        "при создании объявлений с рекламой мобильного приложения")
@Issue("https://st.yandex-team.ru/DIRECT-44779")
@Tag(TagDictionary.TRUNK)
public class AddMobileAppAdDatabaseValuesTest {

    private static final String LOGIN = AdsLogins.CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static int shard;
    private static long adGroupId;

    @BeforeClass
    public static void prepareData() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);

        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);

        api.userSteps
                .getDirectJooqDbSteps()
                .useShard(shard)
                .adGroupsSteps()
                .updatePhrases(new PhrasesRecord()
                        .setPid(adGroupId)
                        .setAdgroupType(PhrasesAdgroupType.mobile_content));
    }

    @Before
    public void addUnitsToUser() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    @Description("Проверка правильности сохранения заголовка, описания и ссылки объявления" +
            " с рекламой мобильного приложения")
    public void checkTitleAndTextOfMobileAppAd() {
        String title = "Некоторое название";
        String text = "Некоторое описание мобильного приложения";
        String trackingUrl = "http://appmetrika.yandex.ru?pid=yandexdirect_int&clickid={logid}";

        long adId = api.userSteps.adsSteps().addAd(new AdAddItemMap()
                .withAdGroupId(adGroupId)
                .withMobileAppAd(new MobileAppAdAddMap()
                        .defaultMobileAppAd()
                        .withTitle(title)
                        .withText(text)
                        .withTrackingUrl(trackingUrl)));

        BannersRecord actualBanner = api.userSteps
                .getDirectJooqDbSteps()
                .useShard(shard)
                .bannersSteps()
                .getBanner(adId);

        assertThat("заголовок, описание и ссылка объявления сохранились верно",
                actualBanner,
                allOf(
                        hasProperty("title", equalTo(title)),
                        hasProperty("body", equalTo(text)),
                        hasProperty("href", equalTo(trackingUrl))));
    }

    @Test
    @Description("Проверка правильности сохранения аттрибутов объявления: аттрибуты включены")
    public void checkEnabledFeaturesOfMobileAppAd() {
        long adId = api.userSteps.adsSteps().addAd(new AdAddItemMap()
                .withAdGroupId(adGroupId)
                .withMobileAppAd(new MobileAppAdAddMap()
                        .defaultMobileAppAd()
                        .withFeatures(
                                new MobileAppAdFeatureItemMap()
                                        .withEnabled(YesNoEnum.YES)
                                        .withFeature(MobileAppFeatureEnum.PRICE),
                                new MobileAppAdFeatureItemMap()
                                        .withEnabled(YesNoEnum.YES)
                                        .withFeature(MobileAppFeatureEnum.RATINGS))));

        BannersMobileContentRecord actualMobileContent = api.userSteps
                .getDirectJooqDbSteps()
                .useShard(shard)
                .mobileContentSteps()
                .getBannersMobileContent(adId);

        Assert.assertThat("атрибуты объявления сохранились верно",
                actualMobileContent.getReflectedAttrs(),
                equalTo(String.join(",", ReflectedAttrs.PRICE.toString(), ReflectedAttrs.RATING_VOTES.toString())));
    }

    @Test
    @Description("Проверка правильности сохранения аттрибутов объявления: один аттрибут включен, другой - выключен")
    public void checkDisabledFeaturesOfMobileAppAd() {
        long adId = api.userSteps.adsSteps().addAd(new AdAddItemMap()
                .withAdGroupId(adGroupId)
                .withMobileAppAd(new MobileAppAdAddMap()
                        .defaultMobileAppAd()
                        .withFeatures(
                                new MobileAppAdFeatureItemMap()
                                        .withEnabled(YesNoEnum.YES)
                                        .withFeature(MobileAppFeatureEnum.PRICE),
                                new MobileAppAdFeatureItemMap()
                                        .withEnabled(YesNoEnum.NO)
                                        .withFeature(MobileAppFeatureEnum.RATINGS))));

        BannersMobileContentRecord actualMobileContent = api.userSteps
                .getDirectJooqDbSteps()
                .useShard(shard)
                .mobileContentSteps()
                .getBannersMobileContent(adId);

        Assert.assertThat("атрибуты объявления сохранились верно",
                actualMobileContent.getReflectedAttrs(),
                equalTo(ReflectedAttrs.PRICE.toString()));
    }

    @Test
    @Description("Проверка правильности сохранения действия объявления")
    public void checkActionOfMobileAppAd() {
        long adId = api.userSteps.adsSteps().addAd(new AdAddItemMap()
                .withAdGroupId(adGroupId)
                .withMobileAppAd(new MobileAppAdAddMap()
                        .defaultMobileAppAd()
                        .withAction(MobileAppAdActionEnum.DOWNLOAD)));

        BannersMobileContentRecord actualMobileContent = api.userSteps
                .getDirectJooqDbSteps()
                .useShard(shard)
                .mobileContentSteps()
                .getBannersMobileContent(adId);

        Assert.assertThat("действие объявления сохранились верно",
                actualMobileContent.getPrimaryAction(),
                equalTo(BannersMobileContentPrimaryAction.download));
    }
}
