package ru.yandex.autotests.direct.api.ads.add.textad;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.ads.PriceCurrencyEnum;
import com.yandex.direct.api.v5.ads.PriceQualifierEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.PriceExtensionAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static com.yandex.direct.api.v5.ads.PriceCurrencyEnum.RUB;
import static com.yandex.direct.api.v5.ads.PriceQualifierEnum.FROM;
import static ru.yandex.autotests.directapi.model.api5.general.ExpectedResult.errors;

@Aqua.Test
@Features(AdsFeatures.ADD)
@Description("Проверка работы поля BannerPrice: негативные кейсы")
@Issue("https://st.yandex-team.ru/DIRECT-89499")
@RunWith(Parameterized.class)
public class AddBannerPriceNegativeTest {
    @ClassRule
    public static final ApiSteps api = new ApiSteps().version(104).as(AdsLogins.CLIENT);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    static Long campaignId;
    static Long adGroupId;

    private static final Long ONE = 1000000L;
    private static final Long TEN = 10000000L;

    // Max value = 10^10 to micros
    private static final Long MAX = BigDecimal.TEN.pow(16).longValue();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public String description;
    @Parameterized.Parameter(1)
    public PriceExtensionAddItemMap price;
    @Parameterized.Parameter(2)
    public Api5Error expectedError;
    @Parameterized.Parameter(3)
    public ExpectedResult expectedResult;

    TextAdAddMap textAdAddMap;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        return Arrays.asList(
                new Object[]{"Не указана валюта", price(ONE, TEN, null, FROM),
                        missingField("PriceCurrency"), null},
                new Object[]{"Не указан префикс", price(ONE, TEN, RUB, null),
                        missingField("PriceQualifier"), null},
                new Object[]{"Цена равная нулю", price(0L, TEN, RUB, FROM), null,
                        lessThanMin("PriceExtension.Price", "0")},
                new Object[]{"Цена отрицательная", price(-ONE, TEN, RUB, FROM), null,
                        lessThanMin("PriceExtension.Price", "0")},
                new Object[]{"Старая цена отрицательная", price(ONE, -TEN, RUB, FROM), null,
                        lessThanMin("PriceExtension.OldPrice", "0")},
                new Object[]{"Старая цена меньше новой", price(TEN, ONE, RUB, FROM), null,
                        oldPriceLowerThanNew()},
                new Object[]{"Цена слишком большая", price(BigInteger.TEN.pow(18).longValue(),null, RUB, FROM), null,
                        moreThanMax("PriceExtension.Price", Long.toString(MAX))},
                new Object[]{"Старая цена слишком большая", price(TEN, BigInteger.TEN.pow(18).longValue(), RUB, FROM), null,
                        moreThanMax("PriceExtension.OldPrice", Long.toString(MAX))}
        );
    }

    private static ExpectedResult lessThanMin(String field, String value) {
        return errors(new Notification(5005, Api5ErrorDetailsJava.VALUE_MUST_BE_GREATER_THAN, field, value));
    }

    private static ExpectedResult moreThanMax(String field, String value) {
        return errors(new Notification(5005, Api5ErrorDetailsJava.VALUE_MUST_NOT_BE_GREATER_THAN, field, value));
    }

    private static ExpectedResult oldPriceLowerThanNew() {
        return errors(new Notification(5005, Api5ErrorDetailsJava.BANNER_OLD_PRICE_LOWER_THAN_NEW));
    }

    private static Api5Error missingField(String fieldName) {
        return new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,fieldName);
    }

    private static PriceExtensionAddItemMap price(Long price, Long oldPrice, PriceCurrencyEnum currency,
            PriceQualifierEnum prefix)
    {
        return new PriceExtensionAddItemMap()
                .withPrice(price)
                .withOldPrice(oldPrice)
                .withPriceCurrency(currency)
                .withPriceQualifier(prefix);
    }

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdsLogins.CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(AdsLogins.CLIENT);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
    }

    @Before
    public void initMap() {
        textAdAddMap = new TextAdAddMap()
                .withText(TextAdAddMap.DEFAULT_TEXT)
                .withTitle(TextAdAddMap.DEFAULT_TITLE)
                .withHref(TextAdAddMap.DEFAULT_HREF)
                .withMobile(YesNoEnum.NO)
                .withPriceExtension(price);
    }

    @Test
    public void addWithBannerPriceAndCheckError() {
        //DIRECT-41067
        AddRequestMap addRequestMap = new AddRequestMap()
                .withAds(new AdAddItemMap()
                        .withTextAd(textAdAddMap)
                        .withAdGroupId(adGroupId));

        if (expectedError != null) {
            api.userSteps.adsSteps().expectErrorOnAdsAdd(addRequestMap, expectedError);
        } else {
            api.userSteps.adsSteps().shouldGetResultOnAdd(addRequestMap, expectedResult);
        }
    }
}
