package ru.yandex.autotests.direct.api.ads.add.textad;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.general.YesNoEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.RandomStringUtils.random;
import static org.apache.commons.lang3.RandomStringUtils.randomAlphabetic;
import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.junit.runners.Parameterized.Parameter;
import static org.junit.runners.Parameterized.Parameters;
import static ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult.Common.fieldLengthExceeded;
import static ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult.bothJavaAndPerl;
import static ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult.expect;
import static ru.yandex.autotests.directapi.steps.banners.AdsSteps.MAX_NUMBER_OF_NARROW_CHARACTERS;
import static ru.yandex.autotests.directapi.steps.banners.AdsSteps.NARROW_CHARACTERS;
import static ru.yandex.autotests.directapi.steps.banners.AdsSteps.generateTextOfLength;

/**
 * Created by pavryabov on 14.04.15.
 * https://st.yandex-team.ru/TESTIRT-5093
 */
@Aqua.Test
@Features(AdsFeatures.ADD)
@Description("Проверка валидации поля Text. Негативные сценарии")
@Issue("https://st.yandex-team.ru/DIRECT-34558")
@RunWith(Parameterized.class)
public class AddTextValidationNegativeTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(AdsLogins.CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    static Long campaignId;
    static Long adGroupId;
    TextAdAddMap textAdAddMap;

    @Parameter(0)
    public String description;

    @Parameter(1)
    public String text;

    @Parameter(2)
    public JavaOrPerlExpectedResult expectedResult;

    @Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        String wordWithGreaterThanMaxLength = randomAlphabetic(TextAdAddMap.MAX_WORD_IN_TEXT_LENGTH + 1);
        return Arrays.asList(new Object[][]{
                {"Добавить баннер с пустым текстом", "",
                        bothJavaAndPerl(5000, Api5ErrorDetails.NO_VALUE_IN_FIELD, capitalize(TextAdAddMap.TEXT))},
                {"Добавить баннер с текстом, который состоит из нескольких пробелов", "   ",
                        bothJavaAndPerl(5000, Api5ErrorDetails.NO_VALUE_IN_FIELD, capitalize(TextAdAddMap.TEXT))},
                {"Добавить баннер с текстом, который содержит запрещённые символы", "サムライ",
                        expect(5002,
                                Api5ErrorDetails.USE_SPECIFIED_ALPHABETS_NUMBERS_AND_PUNCTUATION_IN_FIELD,
                                Api5ErrorDetailsJava.USE_SPECIFIED_ALPHABETS_NUMBERS_AND_PUNCTUATION_IN_FIELD_EN,
                                capitalize(TextAdAddMap.TEXT))},
                {"Добавить баннер с текстом, в котором неправильно использована запятая", "word ,word",
                        bothJavaAndPerl(5006,
                                Api5ErrorDetails.IMPROPER_USE_OF_PUNCTUATION_MARKS_IN_THE_FIELD,
                                capitalize(TextAdAddMap.TEXT))},
                {"Добавить баннер с текстом, содержащим слово, превышающее максимальную длину",
                        wordWithGreaterThanMaxLength,
                        bothJavaAndPerl(5006, Api5ErrorDetails.FIELD_CANT_CONTAINS_WORDS_OF_LENGTH,
                                TextAdAddMap.MAX_WORD_IN_TEXT_LENGTH, capitalize(TextAdAddMap.TEXT),
                                wordWithGreaterThanMaxLength)},
                {"Добавить баннер с текстом, в котором количетво широких символов превышает максимально допустимое ",
                        generateTextOfLength(TextAdAddMap.MAX_TEXT_LENGTH + 1, TextAdAddMap.MAX_WORD_IN_TEXT_LENGTH, 0),
                        fieldLengthExceeded(capitalize(TextAdAddMap.TEXT), TextAdAddMap.MAX_TEXT_LENGTH)},
                {"Добавить баннер с текстом, в котором количетво широких символов превышает максимально допустимое, количество узких символов - максимальное",
                        generateTextOfLength(TextAdAddMap.MAX_TEXT_LENGTH + 1, TextAdAddMap.MAX_WORD_IN_TEXT_LENGTH,
                                MAX_NUMBER_OF_NARROW_CHARACTERS),
                        fieldLengthExceeded(capitalize(TextAdAddMap.TEXT), TextAdAddMap.MAX_TEXT_LENGTH)},
                {"Добавить баннер с текстом, в котором количетво узких символов превышает максимально допустимое ",
                        generateTextOfLength(TextAdAddMap.MAX_TEXT_LENGTH, TextAdAddMap.MAX_WORD_IN_TEXT_LENGTH,
                                MAX_NUMBER_OF_NARROW_CHARACTERS + 1),
                        expect(5006,
                                Api5ErrorDetails.MAX_NUMBER_OF_NARROW_CHARACTERS_IN_FIELD_EXCEEDED,
                                Api5ErrorDetailsJava.MAX_NUMBER_OF_NARROW_CHARACTERS_IN_FIELD_EXCEEDED,
                                capitalize(TextAdAddMap.TEXT), MAX_NUMBER_OF_NARROW_CHARACTERS)},
                {"Добавить баннер с текстом, состоящем из узких символов, количество которых превышает максимальное",
                        random(MAX_NUMBER_OF_NARROW_CHARACTERS + 1, NARROW_CHARACTERS),
                        expect(5006,
                                Api5ErrorDetails.MAX_NUMBER_OF_NARROW_CHARACTERS_IN_FIELD_EXCEEDED,
                                Api5ErrorDetailsJava.MAX_NUMBER_OF_NARROW_CHARACTERS_IN_FIELD_EXCEEDED,
                                capitalize(TextAdAddMap.TEXT), MAX_NUMBER_OF_NARROW_CHARACTERS)},
        });
    }

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdsLogins.CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(AdsLogins.CLIENT);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
    }

    @Before
    public void initMap() {
        textAdAddMap = new TextAdAddMap()
                .withText(text)
                .withTitle(TextAdAddMap.DEFAULT_TITLE)
                .withHref(TextAdAddMap.DEFAULT_HREF)
                .withMobile(YesNoEnum.NO);
    }

    @Test
    public void addWithText() {
        //DIRECT-41068
        api.userSteps.adsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAds(new AdAddItemMap()
                                .withTextAd(textAdAddMap)
                                .withAdGroupId(adGroupId)),
                expectedResult);
    }
}
