package ru.yandex.autotests.direct.api.ads.adgroup;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersMinusGeoType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PhrasesStatusbssynced;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.PhrasesRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.model.RegionIDValues;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

@Aqua.Test
@Features({AdsFeatures.SUSPEND, AdsFeatures.RESUME})
@Description("Проверка сброса статуса bsSynced группы при отключении/включении баннера")
@Issue("https://st.yandex-team.ru/DIRECT-73004")
@RunWith(Parameterized.class)
public class CheckAdGroupStatusBsSyncedWhenSuspendOrResumeBannerTest {

    private static DirectJooqDbSteps jooqDbSteps;
    private static Long campaignId;
    private static Long adGroupId;

    private Long adId;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(AdsLogins.CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter
    public String assertMessage;

    @Parameterized.Parameter(value = 1)
    public RegionIDValues bannerMinusGeoToSet;

    @Parameterized.Parameter(value = 2)
    public PhrasesStatusbssynced expectedAdGroupsStatusBsSynced;

    @Parameterized.Parameters(name = "assertMessage = {0}, bannerMinusGeoToSet = {1}, expectedAdGroupsStatusBsSynced = {2}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {"статус группы не меняется т.к. у баннера нет минус-гео",
                        null, PhrasesStatusbssynced.Yes},
                {"статус группы не меняется т.к. у баннера есть минус-гео, но он не влияет на таргенинг группы",
                        RegionIDValues.KAZAKHSTAN, PhrasesStatusbssynced.Yes},
                {"статус группы должен сброситься т.к. у баннера есть минус-гео, который ограничивает таргенинг группы",
                        RegionIDValues.UKRAINE, PhrasesStatusbssynced.No},
        });
    }

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdsLogins.CLIENT);
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(AdsLogins.CLIENT);

        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        adGroupId = api.userSteps.adGroupsSteps().addGroup(new AdGroupAddItemMap()
                .defaultAdGroupAddItem(campaignId)
                .withRegionIds(RegionIDValues.RUSSIA.getId().longValue(), RegionIDValues.UKRAINE.getId().longValue()));
    }

    @Before
    public void createObjects() {
        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(adGroupId, Status.YES);

        adId = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        saveBannerMinusGeo(bannerMinusGeoToSet);
    }


    @Test
    public void checkAdGroupStatusBsSyncedAfterSuspendingBanner() {
        api.userSteps.bannersFakeSteps().makeBannerActive(adId);
        api.userSteps.adsSteps().adsSuspend(adId);

        PhrasesRecord groupParams = jooqDbSteps.adGroupsSteps().getPhrases(adGroupId);
        assertThat("статус bsSynced группы соответствует ожиданиям после отключения баннера: " + assertMessage,
                groupParams.getStatusbssynced(), equalTo(expectedAdGroupsStatusBsSynced));
    }

    @Test
    public void checkAdGroupStatusBsSyncedAfterResumingBanner() {
        api.userSteps.bannersFakeSteps().makeBannersStopped(adId);
        api.userSteps.adsSteps().adsResume(adId);

        PhrasesRecord groupParams = jooqDbSteps.adGroupsSteps().getPhrases(adGroupId);
        assertThat("статус bsSynced группы соответствует ожиданиям после включения баннера: " + assertMessage,
                groupParams.getStatusbssynced(), equalTo(expectedAdGroupsStatusBsSynced));
    }

    private void saveBannerMinusGeo(RegionIDValues region) {
        if (region != null) {
            jooqDbSteps.bannersSteps()
                    .saveBannersMinusGeo(adId, BannersMinusGeoType.current, region.getId().toString());
        }
    }
}
