package ru.yandex.autotests.direct.api.ads.archive;

import java.sql.Timestamp;
import java.time.LocalDateTime;
import java.util.Arrays;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusarch;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusshow;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.enums.AdGroupType;
import ru.yandex.autotests.directapi.enums.BannerType;
import ru.yandex.autotests.directapi.enums.CampaignType;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 22.04.15.
 * https://st.yandex-team.ru/TESTIRT-5211
 */
@Aqua.Test
@Features(AdsFeatures.ARCHIVE)
@Description("Проверка влияния Archive на статусы объявления")
@Issue("https://st.yandex-team.ru/DIRECT-34573")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class CheckFakeStatusesAfterArchiveTest {

    private static final String LOGIN = AdsLogins.CLIENT;
    private static Timestamp lastChange = Timestamp.valueOf(LocalDateTime.now().minusDays(5));

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long campaignId;
    private Long adID;

    @Parameterized.Parameter(0)
    public CampaignType campaignType;

    @Parameterized.Parameter(1)
    public AdGroupType adGroupType;

    @Parameterized.Parameter(2)
    public BannerType bannerType;

    @Parameterized.Parameters(name = "bannerType = {2}")
    public static Iterable<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {CampaignType.TEXT, AdGroupType.BASE, BannerType.TEXT},
                {CampaignType.MOBILE_CONTENT, AdGroupType.MOBILE_CONTENT, BannerType.MOBILE_CONTENT},
                {CampaignType.DYNAMIC, AdGroupType.DYNAMIC, BannerType.DYNAMIC},
        });
    }

    @BeforeClass
    public static void createAdGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createAd() {
        campaignId = api.userSteps.campaignSteps().addDefaultCampaign(campaignType);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId, adGroupType);

        adID = api.userSteps.adsSteps().addDefaultAd(adGroupId, bannerType);
        api.userSteps.bannersFakeSteps().makeBannersModerated(adID);
    }

    @Test
    @Description("Проверка изменения значения StatusShow после архивации")
    public void checkStatusShowAfterArchive() {
        BannersRecord adBefore = jooqSteps(LOGIN).bannersSteps().getBanner(adID);
        assumeThat("значение StatusShow = Yes",
                adBefore.getStatusshow(), equalTo(BannersStatusshow.Yes));
        api.userSteps.adsSteps().adsArchive(adID);
        BannersRecord adAfter = jooqSteps(LOGIN).bannersSteps().getBanner(adID);
        assertThat("значение StatusShow изменилось на No",
                adAfter.getStatusshow(), equalTo(BannersStatusshow.No));
    }

    @Test
    @Description("Проверка изменения значения LastChange после архивации")
    public void checkLastChangeAfterArchive() {
        jooqSteps(LOGIN).bannersSteps().setBannersLastChange(adID, lastChange);
        api.userSteps.adsSteps().adsArchive(adID);
        Timestamp lastChangeAfter = jooqSteps(LOGIN).bannersSteps().getBanner(adID).getLastchange();
        assertThat("значение LastChange изменилось", lastChangeAfter, not(equalTo(lastChange)));
    }

    @Test
    @Description("Проверка изменения значения StatusArch после архивации")
    public void checkStatusArchivedAfterArchive() {
        BannersStatusarch statusBefore = jooqSteps(LOGIN).bannersSteps().getBanner(adID).getStatusarch();
        assumeThat("значение StatusArch = No", statusBefore, equalTo(BannersStatusarch.No));
        api.userSteps.adsSteps().adsArchive(adID);
        BannersStatusarch statusAfter = jooqSteps(LOGIN).bannersSteps().getBanner(adID).getStatusarch();
        assertThat("значение StatusArch изменилось на Yes", statusAfter, equalTo(BannersStatusarch.Yes));
    }


    private static DirectJooqDbSteps jooqSteps(String login) {
        return api.userSteps.getDirectJooqDbSteps().useShardForLogin(login);
    }
}
