package ru.yandex.autotests.direct.api.ads.archive;

import com.yandex.direct.api.v5.campaigns.CampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.CampaignStateGetEnum;
import com.yandex.direct.api.v5.campaigns.GetResponse;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.common.Value;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 16.04.15.
 * https://st.yandex-team.ru/TESTIRT-5211
 */
@Aqua.Test
@Features(AdsFeatures.ARCHIVE)
@Description("Id, которые должны вызывать ошибку в запросе на архивацию объявления")
@Issue("https://st.yandex-team.ru/DIRECT-34573")
@Tag(TagDictionary.TRUNK)
public class IdsNegativeTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(AdsLogins.CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long campaignId;
    private Long id;

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdsLogins.CLIENT);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdsLogins.CLIENT_ELSE);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createAd() {
        api.as(AdsLogins.CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(AdsLogins.CLIENT);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        id = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
    }

    @Test
    @Description("Нулевой идентификатор объявления в запросе")
    public void archiveZeroId() {
        api.userSteps.adsSteps()
                .shouldGetResultOnArchive(new IdsCriteriaMap().withIds(0L),
                        ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AD_NOT_FOUND)));
    }

    @Test
    @Description("Чужой идентификатор объявления в запросе")
    public void archiveNotMyId() {
        api.as(AdsLogins.CLIENT_ELSE);
        api.userSteps.adsSteps()
                .shouldGetResultOnArchive(new IdsCriteriaMap().withIds(id),
                        ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AD_NOT_FOUND)));
    }

    @Test
    @Description("Удалённый идентификатор объявления в запросе")
    public void archiveDeletedId() {
        api.userSteps.adsSteps().adsDelete(id);
        api.userSteps.adsSteps()
                .shouldGetResultOnArchive(new IdsCriteriaMap().withIds(id),
                        ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AD_NOT_FOUND)));
    }

    @Test
    @Description("Идентификатор объявления из удалённой кампании в запросе")
    public void archiveIdFromDeletedCampaign() {
        api.userSteps.campaignSteps().campaignsDelete(campaignId);
        api.userSteps.adsSteps()
                .shouldGetResultOnArchive(new IdsCriteriaMap().withIds(id),
                        ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AD_NOT_FOUND)));
    }

    @Test
    @Description("Идентификатор объявления из заархивированной кампании в запросе")
    public void archiveIdFromArchivedCampaign() {
        api.userSteps.bannersFakeSteps().makeBannersModerated(id);
        api.userSteps.archiveCampaign(campaignId);
        GetResponse response = api.userSteps.campaignSteps().campaignsGet(new GetRequestMap()
                .withSelectionCriteria(new CampaignsSelectionCriteriaMap().withIds(campaignId))
                .withFieldNames(CampaignFieldEnum.ID, CampaignFieldEnum.STATE));
        CampaignStateGetEnum state = response.getCampaigns().get(0).getState();
        assumeThat("кампания заархивировалась", state, equalTo(CampaignStateGetEnum.ARCHIVED));
        api.userSteps.adsSteps()
                .shouldGetResultOnArchive(new IdsCriteriaMap().withIds(id),
                        ExpectedResult.errors(new Notification(8300,
                                Api5ErrorDetails.UNABLE_UPDATE_ARCHIVED_CAMPAGN)));
    }

    @Test
    @Description("Несуществующий идентификатор объявления в запросе")
    public void archiveNonExistentId() {
        api.userSteps.adsSteps()
                .shouldGetResultOnArchive(new IdsCriteriaMap().withIds(id + 500000),
                        ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AD_NOT_FOUND)));
    }

    @Test
    @Description("Отрицательный идентификатор объявления в запросе")
    public void archiveNegativeId() {
        api.userSteps.adsSteps()
                .shouldGetResultOnArchive(new IdsCriteriaMap().withIds(-id),
                        ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AD_NOT_FOUND)));
    }

    @Test
    @Description("Идентификатор объявления-черновика в запросе")
    public void archiveDraft() {
        api.userSteps.adsSteps()
                .shouldGetResultOnArchive(new IdsCriteriaMap().withIds(id),
                        ExpectedResult.errors(new Notification(8300, Api5ErrorDetails.AD_IS_DRAFT_AND_CANT_BE_ARCHIVED)));
    }

    @Test
    @Description("Идентификатор активного объявления в запросе")
    public void archiveActiveAd() {
        api.userSteps.bannersFakeSteps().makeBannerActive(id);
        BannerFakeInfo bannerFakeInfo = api.userSteps.bannersFakeSteps().getBannerParams(id);
        assumeThat("объявление является активным", bannerFakeInfo.getStatusActive(), equalTo(Value.YES));

        api.userSteps.adsSteps()
                .shouldGetResultOnArchive(new IdsCriteriaMap().withIds(id),
                        ExpectedResult.errors(new Notification(8300, Api5ErrorDetails.CANT_ARCHIVE_ACTIVE_AD)));
    }
}
