package ru.yandex.autotests.direct.api.ads.delete;

import java.util.List;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.ads.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 14.05.15.
 * https://st.yandex-team.ru/TESTIRT-5497
 */
@Aqua.Test
@Features(AdsFeatures.DELETE)
@Description("Проверка правильности списания баллов при удалении объявления")
@Issue("https://st.yandex-team.ru/DIRECT-39274")
public class AdsDeleteUnitsTest {
    private static String client = AdsLogins.DELETE_UNITS;

    private static final int CALL_COST = 10;
    private static final int AD_COST = 0;
    private static final int DEFAULT_ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR = 50;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().version(104).as(client);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long adId1;
    private Long adId2;

    @Before
    @Step("Подготовка данных для теста")
    public void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        List<Long> adIds = api.userSteps.adsSteps().addDefaultTextAds(adGroupId, 2);
        adId1 = adIds.get(0);
        adId2 = adIds.get(1);
    }

    @Test
    @Description("Удалить одно объявление")
    public void deleteOneAd() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.adsSteps().adsDelete(adId1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + AD_COST));
    }

    @Test
    @Description("Удалить два объявления")
    public void deleteTwoAds() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.adsSteps().adsDelete(adId1, adId2);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + 2 * AD_COST));
    }

    @Test
    @Description("Запрос на удаление объявления с ошибкой")
    public void deleteOneAdWithError() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.adsSteps().adsDelete(-adId1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + DEFAULT_ERROR_COST));
    }

    @Test
    @Description("Запрос на удаление двух объявлений с одной ошибкой")
    public void deleteTwoAdsWithOneError() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.adsSteps().adsDelete(adId1, -adId1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + DEFAULT_ERROR_COST + AD_COST));
    }

    @Test
    @Description("Запрос на удаление объявлений с двумя ошибками")
    public void deleteTwoAdsWithTwoErrors() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.adsSteps().adsDelete(-adId1, -adId1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + 2 * DEFAULT_ERROR_COST));
    }

    @Test
    @Description("Невалидный запрос на удаление объявления")
    public void deleteWithInvalidRequest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.adsSteps().expectErrorOnAdsDelete(
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        path(IdsCriteriaMap.IDS)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(COMMON_REQUEST_ERROR));
    }

    @Test
    @Description("Запрос на удаление объявления, когда у клиента отсутствуют баллы")
    public void callAdsDeleteWithZeroUnits() {
        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(client, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assumeThat("у клиента не осталось баллов", units, equalTo(0));
        api.userSteps.adsSteps().expectErrorOnAdsDelete(new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap().withIds(adId1)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION)
        );
    }
}
