package ru.yandex.autotests.direct.api.ads.delete;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AccessRightsErrorHelper;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.ads.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

/**
 * Created by pavryabov on 30.11.15.
 * https://st.yandex-team.ru/TESTIRT-7862
 */
@Aqua.Test
@Features(AdsFeatures.DELETE)
@Description("Проверка удаления динамических текстовых объявлений от имени разных ролей")
@RunWith(Parameterized.class)
@Issue("https://st.yandex-team.ru/DIRECT-43825")
public class DeleteDynamicTextAdRolesTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public String apiUserLogin;

    @Parameterized.Parameter(2)
    public String login;

    @Parameterized.Parameter(3)
    public String loginToCreateAd;

    @Parameterized.Parameter(4)
    public ExpectedResult expectedResult;

    @Parameterized.Parameter(5)
    public Api5Error expectedJavaError;

    private long adID;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {"Удаление объявления от имени менеджера", Logins.MANAGER_DEFAULT, AdsLogins.SERV_CLIENT,
                        AdsLogins.MANAGER_DEFAULT,
                        ExpectedResult.success(),
                        null},

                {"Удаление объявления от имени вешальщика", Logins.PLACER, AdsLogins.CLIENT,
                        AdsLogins.CLIENT,
                        ExpectedResult.errors(new Notification(54, Api5ErrorDetails.NO_RIGHTS_TO_WRITE)),
                        new Api5Error(54)},

                {"Удаление объявления от имени саппорта", Logins.SUPPORT, AdsLogins.CLIENT,
                        AdsLogins.CLIENT,
                        ExpectedResult.success(),
                        null},

                {"Удаление объявления от имени супера", Logins.SUPER_LOGIN, AdsLogins.CLIENT,
                        AdsLogins.CLIENT,
                        ExpectedResult.success(),
                        null},

                {"Удаление объявления от имени медиапланера", Logins.MEDIA, AdsLogins.CLIENT,
                        AdsLogins.CLIENT,
                        ExpectedResult.errors(new Notification(54, Api5ErrorDetails.NO_RIGHTS_TO_WRITE)),
                        new Api5Error(54)},

                {"Удаление объявления от имени суперридера", Logins.SUPER_READER, AdsLogins.CLIENT,
                        AdsLogins.CLIENT,
                        ExpectedResult.errors(new Notification(54, Api5ErrorDetails.NO_RIGHTS_TO_WRITE)),
                        new Api5Error(54)},

                {"Удаление объявления от имени клиента для самого себя", AdsLogins.CLIENT, AdsLogins.CLIENT,
                        AdsLogins.CLIENT,
                        ExpectedResult.success(),
                        null},

                {"Удаление объявления от имени представителя клиента для клиента", AdsLogins.CLIENT_REP,
                        AdsLogins.CLIENT,
                        AdsLogins.CLIENT,
                        ExpectedResult.success(),
                        null},

                {"Удаление объявления от имени агенства для субклиента", AdsLogins.AGENCY, AdsLogins.SUBCLIENT,
                        AdsLogins.AGENCY,
                        ExpectedResult.success(),
                        null},

                {"Удаление объявления от имени представителя агенства для субклиента", AdsLogins.AGENCY_REP,
                        AdsLogins.SUBCLIENT,
                        AdsLogins.AGENCY_REP,
                        ExpectedResult.success(),
                        null},

                {"Удаление объявления от имени субклиента", AdsLogins.SUBCLIENT, AdsLogins.SUBCLIENT,
                        AdsLogins.AGENCY,
                        ExpectedResult.errors(new Notification(54, Api5ErrorDetails.NO_RIGHTS_TO_WRITE)),
                        new Api5Error(54)}
        });
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createAd() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(apiUserLogin);

        api.as(loginToCreateAd);
        Long campaignID = api.userSteps.campaignSteps().addDefaultDynamicTextCampaign(login);
        Long adGroupID = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(campaignID, login);
        adID = api.userSteps.adsSteps().addDefaultDynamicTextAd(adGroupID, login);
    }

    @Test
    public void requestUsingLogin() {
        api.as(apiUserLogin);
        AccessRightsErrorHelper.checkCatchingJavaError54(expectedJavaError, () -> api.userSteps.adsSteps()
                .shouldGetResultOnDelete(new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(adID)), login, expectedResult));
    }
}
