package ru.yandex.autotests.direct.api.ads.get;

import java.util.Arrays;
import java.util.List;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.GetResponse;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.ads.AdGetItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.LimitOffsetMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.extractProperty;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by pavryabov on 23.05.15.
 * https://st.yandex-team.ru/TESTIRT-5331
 */
@Aqua.Test
@Features(AdsFeatures.GET)
@Description("Проверка пагинации")
@Issue("https://st.yandex-team.ru/DIRECT-34556")
@Tag(TagDictionary.TRUNK)
public class PagerTest {

    private static String client = AdsLogins.CLIENT;
    private static String clientAnother = AdsLogins.CLIENT_ELSE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    static Long campaignId;
    private static Long id1;
    private static Long id2;
    private static Long id3;
    private static Long id4;

    private static Long campaignIdAnother;

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(clientAnother);

        campaignId = api.as(client).userSteps.campaignSteps().addDefaultTextCampaign(client);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        campaignIdAnother = api.as(clientAnother).userSteps.campaignSteps().addDefaultTextCampaign(clientAnother);
        Long adGroupIdAnother = api.userSteps.adGroupsSteps().addDefaultGroup(campaignIdAnother);

        id1 = api.as(client).userSteps.adsSteps().addDefaultTextAd(adGroupId);
        api.as(clientAnother).userSteps.adsSteps().addDefaultTextAd(adGroupIdAnother);
        id2 = api.as(client).userSteps.adsSteps().addDefaultTextAd(adGroupId);
        id3 = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        id4 = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
    }

    @Test
    public void getHead() {
        //DIRECT-42163
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                .withFieldNames(AdFieldEnum.ID)
                .withPage(new LimitOffsetMap()
                        .withLimit(2L)
                        .withOffset(0L)));
        List<Long> gotIds = extractProperty(getResponse.getAds(), AdGetItemMap.ID);
        assertThat("вернулся правильный набор объявлений", gotIds, equalTo(Arrays.asList(id1, id2)));
    }

    @Test
    public void getBody() {
        //DIRECT-42163
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                .withFieldNames(AdFieldEnum.ID)
                .withPage(new LimitOffsetMap()
                        .withLimit(2L)
                        .withOffset(1L)));
        List<Long> gotIds = extractProperty(getResponse.getAds(), AdGetItemMap.ID);
        assertThat("вернулся правильный набор объявлений", gotIds, equalTo(Arrays.asList(id2, id3)));
    }

    @Test
    public void getTail() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                .withFieldNames(AdFieldEnum.ID)
                .withPage(new LimitOffsetMap()
                        .withLimit(2L)
                        .withOffset(2L)));
        List<Long> gotIds = extractProperty(getResponse.getAds(), AdGetItemMap.ID);
        assertThat("вернулся правильный набор объявлений", gotIds, equalTo(Arrays.asList(id3, id4)));
    }

    @Test
    public void getPartOfTail() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                .withFieldNames(AdFieldEnum.ID)
                .withPage(new LimitOffsetMap()
                        .withLimit(2L)
                        .withOffset(3L)));
        List<Long> gotIds = extractProperty(getResponse.getAds(), AdGetItemMap.ID);
        assertThat("вернулся правильный набор объявлений", gotIds, equalTo(Arrays.asList(id4)));
    }

    @Test
    public void getOverTail() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                .withFieldNames(AdFieldEnum.ID)
                .withPage(new LimitOffsetMap()
                        .withLimit(2L)
                        .withOffset(4L)));
        assertThat("вернулся правильный набор объявлений", getResponse.getAds(), hasSize(0));
    }

    @Test
    public void onlyLimit() {
        //DIRECT-42163
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                .withFieldNames(AdFieldEnum.ID)
                .withPage(new LimitOffsetMap()
                        .withLimit(2L)));
        List<Long> gotIds = extractProperty(getResponse.getAds(), AdGetItemMap.ID);
        assertThat("вернулся правильный набор объявлений", gotIds, equalTo(Arrays.asList(id1, id2)));
    }

    @Test
    public void onlyOffset() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                .withFieldNames(AdFieldEnum.ID)
                .withPage(new LimitOffsetMap()
                        .withOffset(2L)));
        List<Long> gotIds = extractProperty(getResponse.getAds(), AdGetItemMap.ID);
        assertThat("вернулся правильный набор объявлений", gotIds, equalTo(Arrays.asList(id3, id4)));
    }

    @Test
    public void zeroLimit() {
        api.userSteps.adsSteps().expectErrorOnAdsGet(new GetRequestMap()
                        .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                        .withFieldNames(AdFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(0L)),
                new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0));
    }

    @Test
    public void negativeLimit() {
        api.userSteps.adsSteps().expectErrorOnAdsGet(new GetRequestMap()
                        .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                        .withFieldNames(AdFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(-1L)),
                new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0));
    }

    @Test
    public void maxLimit() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                .withFieldNames(AdFieldEnum.ID)
                .withPage(new LimitOffsetMap()
                        .withLimit(LimitOffsetMap.LIMIT_MAX_LENGTH)));
        List<Long> gotIds = extractProperty(getResponse.getAds(), AdGetItemMap.ID);
        assertThat("вернулся правильный набор объявлений", gotIds, equalTo(Arrays.asList(id1, id2, id3, id4)));
    }

    @Test
    public void overMaxLimit() {
        api.userSteps.adsSteps().expectErrorOnAdsGet(new GetRequestMap()
                        .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                        .withFieldNames(AdFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withLimit(LimitOffsetMap.LIMIT_MAX_LENGTH + 1)),
                new Api5Error(4002, Api5ErrorDetails.LIMIT_CANNOT_BE_MORE, LimitOffsetMap.LIMIT_MAX_LENGTH));
    }

    @Test
    public void negativeOffset() {
        api.userSteps.adsSteps().expectErrorOnAdsGet(new GetRequestMap()
                        .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                        .withFieldNames(AdFieldEnum.ID)
                        .withPage(new LimitOffsetMap()
                                .withOffset(-1L)),
                new Api5Error(4002, Api5ErrorDetails.OFFSET_MUST_BE_MORE_OR_EQUAL_THAN_0));
    }

    @Test
    public void checkLimitedByIfLimitDidntWork() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                .withFieldNames(AdFieldEnum.ID)
                .withPage(new LimitOffsetMap()
                        .withLimit(4L)));
        assertThat("значение LimitedBy верно", getResponse.getLimitedBy(), nullValue());
    }

    @Test
    public void checkLimitedByIfLimitWorked() {
        //DIRECT-42266
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                .withFieldNames(AdFieldEnum.ID)
                .withPage(new LimitOffsetMap()
                        .withLimit(2L)));
        assertThat("значение LimitedBy верно", getResponse.getLimitedBy(), equalTo(2L));
    }

    @Test
    public void checkLimitedByIfLimitWorkedWithOffset() {
        //DIRECT-44575
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId))
                .withFieldNames(AdFieldEnum.ID)
                .withPage(new LimitOffsetMap()
                        .withOffset(1L)
                        .withLimit(2L)));
        assertThat("значение LimitedBy верно", getResponse.getLimitedBy(), equalTo(3L));
    }

    @Test
    public void useLimitAndNotOwnedCampaign() {
        //DIRECT-43661
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignId, campaignIdAnother))
                .withFieldNames(AdFieldEnum.ID)
                .withPage(new LimitOffsetMap()
                        .withLimit(2L)));
        List<Long> gotIds = extractProperty(getResponse.getAds(), AdGetItemMap.ID);
        assertThat("вернулся правильный набор объявлений", gotIds, equalTo(Arrays.asList(id1, id2)));
    }
}
