package ru.yandex.autotests.direct.api.ads.get;

import java.util.Arrays;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.AdGetItem;
import com.yandex.direct.api.v5.ads.GetResponse;
import com.yandex.direct.api.v5.ads.TextAdFieldEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdGetItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdGetMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang.StringUtils.capitalize;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanEquivalentV5;
import static ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error.Common.maxIdsInSelectionCriteria;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by pavryabov on 28.04.15.
 * https://st.yandex-team.ru/TESTIRT-5331
 */
@Aqua.Test
@Features(AdsFeatures.GET)
@Description("Проверка фильтрации по SitelinkSetIds")
@Issue("https://st.yandex-team.ru/DIRECT-34556")
@Tag(TagDictionary.TRUNK)
public class SitelinkSetIdsTest {

    private static String client = AdsLogins.CLIENT;
    private static String otherClient = AdsLogins.CLIENT_ELSE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    Long campaignId;
    Long adGroupId;
    Long adId;
    private Long sitelinkSetId;

    @Before
    public void initTest() {
        api.as(client);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        sitelinkSetId = api.userSteps.sitelinksSteps().addDefaultSet(1);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        adId = api.userSteps.adsSteps().addAd(new AdAddItemMap().withTextAd(
                new TextAdAddMap().defaultTextAd().withSitelinkSetId(sitelinkSetId)).withAdGroupId(adGroupId));
    }

    @Test
    public void getOneSitelinksSetId() {
        //DIRECT-41586
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withSitelinkSetIds(sitelinkSetId))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.SITELINK_SET_ID));
        assertThat("вернулось правильное объявление", getResponse.getAds(), beanEquivalentV5(
                Arrays.asList((AdGetItem) new AdGetItemMap().withId(adId)
                        .withTextAd(new TextAdGetMap().withSitelinkSetId(sitelinkSetId)).getBean())));
    }

    @Test
    public void getTwoSitelinkSetIds() {
        Long sitelinkSetIdElse = api.userSteps.sitelinksSteps().addDefaultSet(1);
        Long adIdElse = api.userSteps.adsSteps().addAd(new AdAddItemMap().withTextAd(
                new TextAdAddMap().defaultTextAd().withSitelinkSetId(sitelinkSetIdElse)).withAdGroupId(adGroupId));
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withSitelinkSetIds(sitelinkSetId, sitelinkSetIdElse))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.SITELINK_SET_ID));
        assertThat("вернулись правильные объявления", getResponse.getAds(), beanEquivalentV5(
                Arrays.asList((AdGetItem) new AdGetItemMap().withId(adId)
                                .withTextAd(new TextAdGetMap().withSitelinkSetId(sitelinkSetId)).getBean(),
                        (AdGetItem) new AdGetItemMap().withId(adIdElse)
                                .withTextAd(new TextAdGetMap().withSitelinkSetId(sitelinkSetIdElse)).getBean())));
    }

    @Test
    public void getOneSitelinkSetIdTwoTimes() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withSitelinkSetIds(sitelinkSetId, sitelinkSetId))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.SITELINK_SET_ID));
        assertThat("вернулось правильное объявление", getResponse.getAds(), beanEquivalentV5(
                Arrays.asList((AdGetItem) new AdGetItemMap().withId(adId)
                        .withTextAd(new TextAdGetMap().withSitelinkSetId(sitelinkSetId)).getBean())));
    }

    @Test
    public void getWithMaxLengthOfSitelinkSetIds() {
        Long[] sitelinkSetIds = new Long[AdsSelectionCriteriaMap.SITELINK_SET_IDS_MAX_LENGTH];
        Arrays.fill(sitelinkSetIds, sitelinkSetId);
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withSitelinkSetIds(sitelinkSetIds))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.SITELINK_SET_ID));
        assertThat("вернулось правильное объявление", getResponse.getAds(), beanEquivalentV5(
                Arrays.asList((AdGetItem) new AdGetItemMap().withId(adId)
                        .withTextAd(new TextAdGetMap().withSitelinkSetId(sitelinkSetId)).getBean())));
    }

    @Test
    public void getWithOverMaxLengthOfSitelinkSetIds() {
        Long[] sitelinkSetIds = new Long[AdsSelectionCriteriaMap.SITELINK_SET_IDS_MAX_LENGTH + 1];
        Arrays.fill(sitelinkSetIds, sitelinkSetId);
        api.userSteps.adsSteps().expectErrorOnAdsGet(
                new GetRequestMap()
                        .withSelectionCriteria(new AdsSelectionCriteriaMap()
                                .withCampaignIds(campaignId)
                                .withSitelinkSetIds(sitelinkSetIds))
                        .withFieldNames(AdFieldEnum.ID),
                maxIdsInSelectionCriteria(
                        capitalize(AdsSelectionCriteriaMap.SITELINK_SET_IDS),
                        AdsSelectionCriteriaMap.SITELINK_SET_IDS_MAX_LENGTH));
    }

    @Test
    public void getNonExistentSitelinkSetId() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withSitelinkSetIds(sitelinkSetId + 10000))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }

    @Test
    public void getZeroSitelinkSetId() {
        //DIRECT-41578
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withSitelinkSetIds(0L))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }

    @Test
    public void getNegativeSitelinkSetId() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withSitelinkSetIds(-sitelinkSetId))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }

    @Test
    public void getSitelinkSetIdWithDeletedAd() {
        api.userSteps.adsSteps().adsDelete(adId);
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withSitelinkSetIds(sitelinkSetId))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }

    @Test
    public void getUnbindedSitelinkSetId() {
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap().withAds(
                new AdUpdateItemMap().withId(adId).withTextAd(new TextAdUpdateMap().withSitelinkSetId(null))
        ));
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withSitelinkSetIds(sitelinkSetId))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }

    @Test
    public void getNotMySitelinkSetId() {
        api.as(otherClient);
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withSitelinkSetIds(sitelinkSetId))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }

    @Test
    public void getSitelinkSetIdWithTwoAds() {
        Long adIdElse = api.userSteps.adsSteps().addAd(new AdAddItemMap().withTextAd(
                new TextAdAddMap().defaultTextAd().withSitelinkSetId(sitelinkSetId)).withAdGroupId(adGroupId));
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withSitelinkSetIds(sitelinkSetId))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.SITELINK_SET_ID));
        assertThat("вернулись правильные объявления", getResponse.getAds(), beanEquivalentV5(
                Arrays.asList((AdGetItem) new AdGetItemMap().withId(adId)
                                .withTextAd(new TextAdGetMap().withSitelinkSetId(sitelinkSetId)).getBean(),
                        (AdGetItem) new AdGetItemMap().withId(adIdElse)
                                .withTextAd(new TextAdGetMap().withSitelinkSetId(sitelinkSetId)).getBean())));
    }
}
