package ru.yandex.autotests.direct.api.ads.get;

import java.util.Arrays;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.AdGetItem;
import com.yandex.direct.api.v5.ads.AdStateSelectionEnum;
import com.yandex.direct.api.v5.ads.GetResponse;
import com.yandex.direct.api.v5.general.StateEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.enums.AdGroupType;
import ru.yandex.autotests.directapi.enums.BannerType;
import ru.yandex.autotests.directapi.enums.CampaignType;
import ru.yandex.autotests.directapi.model.api5.ads.AdGetItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by pavryabov on 17.04.15.
 * https://st.yandex-team.ru/TESTIRT-5226
 */
@Aqua.Test
@Features(AdsFeatures.GET)
@Description("Проверка статуса ARCHIVED в ответе Ads.get")
@Issue("https://st.yandex-team.ru/DIRECT-34556")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class StateArchiveTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(AdsLogins.CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    Long campaignId;
    Long adId;

    @Parameterized.Parameter(0)
    public CampaignType campaignType;

    @Parameterized.Parameter(1)
    public AdGroupType adGroupType;

    @Parameterized.Parameter(2)
    public BannerType bannerType;

    @Parameterized.Parameters(name = "bannerType = {2}")
    public static Iterable<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {CampaignType.TEXT, AdGroupType.BASE, BannerType.TEXT},
                {CampaignType.MOBILE_CONTENT, AdGroupType.MOBILE_CONTENT, BannerType.MOBILE_CONTENT},
                {CampaignType.DYNAMIC, AdGroupType.DYNAMIC, BannerType.DYNAMIC},
        });
    }

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdsLogins.CLIENT);
    }

    @Before
    public void createObjects() {
        campaignId = api.userSteps.campaignSteps().addDefaultCampaign(campaignType);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId, adGroupType);
        adId = api.userSteps.adsSteps().addDefaultAd(adGroupId, bannerType);
    }

    @Test
    public void getArchivedAd() {
        api.userSteps.bannersFakeSteps().makeBannersModerated(adId);
        api.userSteps.adsSteps().adsArchive(adId);
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withIds(adId))
                .withFieldNames(AdFieldEnum.ID, AdFieldEnum.STATE));
        assertThat("вернулся правильный State", getResponse.getAds(), beanDifferV5(
                Arrays.asList((AdGetItem) new AdGetItemMap().withId(adId).withState(StateEnum.ARCHIVED).getBean())));
    }

    @Test
    public void getArchivedAdWithFilterByOtherState() {
        //DIRECT-47140
        api.userSteps.bannersFakeSteps().makeBannersModerated(adId);
        api.userSteps.adsSteps().adsArchive(adId);
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(adId)
                        .withStates(
                                AdStateSelectionEnum.ON,
                                AdStateSelectionEnum.OFF,
                                AdStateSelectionEnum.SUSPENDED,
                                AdStateSelectionEnum.OFF_BY_MONITORING))
                .withFieldNames(AdFieldEnum.ID, AdFieldEnum.STATE));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }

    @Test
    public void getAdFromArchivedCampaign() {
        //DIRECT-42024
        api.userSteps.campaignSteps().campaignsSuspend(campaignId);
        api.userSteps.campaignSteps().campaignsArchive(campaignId);
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withIds(adId))
                .withFieldNames(AdFieldEnum.ID, AdFieldEnum.STATE));
        assertThat("вернулся правильный State", getResponse.getAds(), beanDifferV5(
                Arrays.asList((AdGetItem) new AdGetItemMap().withId(adId).withState(StateEnum.ARCHIVED).getBean())));
    }

    @Test
    public void getAdFromArchivedCampaignByOtherState() {
        //DIRECT-42024
        api.userSteps.campaignSteps().campaignsSuspend(campaignId);
        api.userSteps.campaignSteps().campaignsArchive(campaignId);
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(adId)
                        .withStates(
                                AdStateSelectionEnum.ON,
                                AdStateSelectionEnum.OFF,
                                AdStateSelectionEnum.SUSPENDED,
                                AdStateSelectionEnum.OFF_BY_MONITORING
                        ))
                .withFieldNames(AdFieldEnum.ID, AdFieldEnum.STATE));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }
}
