package ru.yandex.autotests.direct.api.ads.get;

import java.util.Arrays;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.AdGetItem;
import com.yandex.direct.api.v5.ads.GetResponse;
import com.yandex.direct.api.v5.ads.TextAdFieldEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdGetItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdGetMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang.StringUtils.capitalize;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanEquivalentV5;
import static ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error.Common.maxIdsInSelectionCriteria;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by pavryabov on 28.04.15.
 * https://st.yandex-team.ru/TESTIRT-5331
 */
@Aqua.Test
@Features(AdsFeatures.GET)
@Description("Проверка фильтрации по VCardIds")
@Issue("https://st.yandex-team.ru/DIRECT-34556")
@Tag(TagDictionary.TRUNK)
public class VCardIdsTest {

    private static String client = AdsLogins.CLIENT;
    private static String otherClient = AdsLogins.CLIENT_ELSE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    Long campaignId;
    Long adGroupId;
    Long adId;
    Long vCardId;

    @Before
    public void initTest() {
        api.as(client);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        vCardId = api.userSteps.vCardsSteps().addDefaultVCard(campaignId);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        adId = api.userSteps.adsSteps().addAd(new AdAddItemMap().withTextAd(
                new TextAdAddMap().defaultTextAd().withVCardId(vCardId)).withAdGroupId(adGroupId));
    }

    @Test
    public void getOneVCardId() {
        //DIRECT-41586
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withVCardIds(vCardId))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.V_CARD_ID));
        assertThat("вернулось правильное объявление", getResponse.getAds(), beanEquivalentV5(
                Arrays.asList((AdGetItem) new AdGetItemMap().withId(adId)
                        .withTextAd(new TextAdGetMap().withVCardId(vCardId)).getBean())));
    }

    @Test
    public void getTwoVCardIds() {
        Long vCardIdElse = api.userSteps.vCardsSteps().addDefaultVCard(campaignId);
        Long adIdElse = api.userSteps.adsSteps().addAd(new AdAddItemMap().withTextAd(
                new TextAdAddMap().defaultTextAd().withVCardId(vCardIdElse)).withAdGroupId(adGroupId));
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withVCardIds(vCardId, vCardIdElse))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.V_CARD_ID));
        assertThat("вернулись правильные объявления", getResponse.getAds(), beanEquivalentV5(
                Arrays.asList((AdGetItem) new AdGetItemMap().withId(adId)
                                .withTextAd(new TextAdGetMap().withVCardId(vCardId)).getBean(),
                        (AdGetItem) new AdGetItemMap().withId(adIdElse)
                                .withTextAd(new TextAdGetMap().withVCardId(vCardIdElse)).getBean())));
    }

    @Test
    public void getOneVCardIdTwoTimes() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withVCardIds(vCardId, vCardId))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.V_CARD_ID));
        assertThat("вернулось правильное объявление", getResponse.getAds(), beanEquivalentV5(
                Arrays.asList((AdGetItem) new AdGetItemMap().withId(adId)
                        .withTextAd(new TextAdGetMap().withVCardId(vCardId)).getBean())));
    }

    @Test
    public void getWithMaxLengthOfVCardIds() {
        Long[] vCardIds = new Long[AdsSelectionCriteriaMap.VCARD_IDS_MAX_LENGTH];
        Arrays.fill(vCardIds, vCardId);
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withVCardIds(vCardIds))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.V_CARD_ID));
        assertThat("вернулось правильное объявление", getResponse.getAds(), beanEquivalentV5(
                Arrays.asList((AdGetItem) new AdGetItemMap().withId(adId)
                        .withTextAd(new TextAdGetMap().withVCardId(vCardId)).getBean())));
    }

    @Test
    public void getWithOverMaxLengthOfVCardIds() {
        Long[] vCardIds = new Long[AdsSelectionCriteriaMap.VCARD_IDS_MAX_LENGTH + 1];
        Arrays.fill(vCardIds, vCardId);
        api.userSteps.adsSteps().expectErrorOnAdsGet(
                new GetRequestMap()
                        .withSelectionCriteria(new AdsSelectionCriteriaMap()
                                .withCampaignIds(campaignId)
                                .withVCardIds(vCardIds))
                        .withFieldNames(AdFieldEnum.ID),
                maxIdsInSelectionCriteria(
                        capitalize(AdsSelectionCriteriaMap.VCARD_IDS),
                        AdsSelectionCriteriaMap.VCARD_IDS_MAX_LENGTH));
    }

    @Test
    public void getNonExistentVCardId() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withVCardIds(vCardId + 10000))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }

    @Test
    public void getZeroVCardId() {
        //DIRECT-41578
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withVCardIds(0L))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }

    @Test
    public void getNegativeVCardId() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withVCardIds(-vCardId))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }

    @Test
    public void getVCardIdWithDeletedAd() {
        api.userSteps.adsSteps().adsDelete(adId);
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withVCardIds(vCardId))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }

    @Test
    public void getUnbindedVCardId() {
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap().withAds(
                new AdUpdateItemMap().withId(adId).withTextAd(new TextAdUpdateMap().withVCardId(null))
        ));
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withVCardIds(vCardId))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }

    @Test
    public void getNotMyVCardId() {
        api.as(otherClient);
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withVCardIds(vCardId))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }

    @Test
    public void getVCardIdWithTwoAds() {
        Long adIdElse = api.userSteps.adsSteps().addAd(new AdAddItemMap().withTextAd(
                new TextAdAddMap().defaultTextAd().withVCardId(vCardId)).withAdGroupId(adGroupId));
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId)
                        .withVCardIds(vCardId))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.V_CARD_ID));
        assertThat("вернулись правильные объявления", getResponse.getAds(), beanEquivalentV5(
                Arrays.asList((AdGetItem) new AdGetItemMap().withId(adId)
                                .withTextAd(new TextAdGetMap().withVCardId(vCardId)).getBean(),
                        (AdGetItem) new AdGetItemMap().withId(adIdElse)
                                .withTextAd(new TextAdGetMap().withVCardId(vCardId)).getBean())));
    }
}
