package ru.yandex.autotests.direct.api.ads.get.mobileappad;

import java.util.Arrays;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.AdGetItem;
import com.yandex.direct.api.v5.ads.AdStateSelectionEnum;
import com.yandex.direct.api.v5.ads.GetResponse;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PhrasesAdgroupType;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.ads.AdGetItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanDiffer;

/**
 * Created by semkagtn on 31.08.15.
 * https://st.yandex-team.ru/TESTIRT-6853
 */
@Aqua.Test
@Features(AdsFeatures.GET)
@Description("Проверка фильтрации по State для объявлений с рекламой мобильных приложений")
@Issue("https://st.yandex-team.ru/DIRECT-44779")
@Tag(TagDictionary.TRUNK)
public class GetMobileAppAdByStateTest {
    private static final String LOGIN = AdsLogins.CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);
    private static DirectJooqDbSteps jooqDbSteps;

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static long onAd;
    private static long offAd;

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        int shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        api.userSteps.campaignFakeSteps().setStatusActive(campaignId, Status.YES);
        long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN);
        jooqDbSteps.adGroupsSteps().setType(adGroupId, PhrasesAdgroupType.mobile_content);

        onAd = api.userSteps.adsSteps().addDefaultMobileAppAd(adGroupId);
        offAd = api.userSteps.adsSteps().addDefaultMobileAppAd(adGroupId);

        api.userSteps.bannersFakeSteps().setStatusShow(onAd, Status.YES);
        api.userSteps.bannersFakeSteps().setStatusActive(onAd, Status.YES);
        api.userSteps.bannersFakeSteps().setStatusShow(offAd, Status.YES);
        api.userSteps.bannersFakeSteps().setStatusActive(offAd, Status.NO);
    }

    @Test
    @Description("Попытка получить включенное объявление, указав в запросе States = [Off]")
    public void getOnAdWithOffStateInRequest() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(onAd)
                        .withStates(AdStateSelectionEnum.OFF))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("не вернулось ни одного объявления", getResponse.getAds(), hasSize(0));
    }

    @Test
    @Description("Получить включенное объявление, указав в запросе States = [On, Off]")
    public void getOnAdWithOnAndOffStatesInRequest() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(onAd)
                        .withStates(AdStateSelectionEnum.ON, AdStateSelectionEnum.OFF))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("вернулось одно объявление", getResponse.getAds(), beanDiffer(
                Arrays.asList((AdGetItem) new AdGetItemMap()
                        .withId(onAd).getBean())));
    }

    @Test
    @Description("Попытка получить включенное и выключенное объявления, указав в запросе States = [On]")
    public void getOnAndOffAdsWithOnStateInRequest() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(onAd, offAd)
                        .withStates(AdStateSelectionEnum.ON))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("вернулся правильный ответ", getResponse.getAds(), beanDiffer(
                Arrays.asList((AdGetItem) new AdGetItemMap()
                        .withId(onAd).getBean())));
    }

    @Test
    @Description("Получить включенные и выключенное объявления, указав в запросе States = [On, Off]")
    public void getOnAndOffAdsWithOnAndOffStatesInRequest() {
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(onAd, offAd)
                        .withStates(AdStateSelectionEnum.ON, AdStateSelectionEnum.OFF))
                .withFieldNames(AdFieldEnum.ID));
        assertThat("вернулся правильный ответ", getResponse.getAds(), beanDiffer(
                Arrays.asList(
                        (AdGetItem) new AdGetItemMap().withId(onAd).getBean(),
                        (AdGetItem) new AdGetItemMap().withId(offAd).getBean())));
    }
}
