package ru.yandex.autotests.direct.api.ads.get.mobileappad;

import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Stream;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.AdGetItem;
import com.yandex.direct.api.v5.ads.MobileAppAdFeatureGetItem;
import com.yandex.direct.api.v5.ads.MobileAppAdFieldEnum;
import com.yandex.direct.api.v5.ads.MobileAppFeatureEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.general.YesNoUnknownEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PhrasesAdgroupType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.AdgroupsMobileContentRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.MobileContentRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.AdgroupsMobileContentUtils;
import ru.yandex.autotests.directapi.darkside.model.MobileContentUtils;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppAdFeatureGetItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppAdFeatureItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by semkagtn on 29.09.15.
 * https://st.yandex-team.ru/TESTIRT-6853
 */
@Aqua.Test
@Features(AdsFeatures.GET)
@Description("Проверка поля IsAvailable в ответе метода GET")
@Issue("https://st.yandex-team.ru/DIRECT-44779")
public class GetMobileAppAdIsAvailableTest {

    private static final String LOGIN = AdsLogins.CLIENT;

    // comparator для того что бы тест не мигал при тестировании perl-версии,
    // в java-версии список фич отдается в отсортированном виде
    private static final Comparator<MobileAppAdFeatureGetItem> comparator =
            Comparator.comparing(e -> e.getFeature().name(), String.CASE_INSENSITIVE_ORDER);

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);
    private static DirectJooqDbSteps jooqDbSteps;

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static String clientId;
    private static Long campaignId;

    @Rule
    public Trashman trasher = new Trashman(api);
    private long adGroupId;
    private long adId;

    @BeforeClass
    public static void prepareData() {
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN);
        clientId = api.userSteps.clientFakeSteps().getClientData(LOGIN).getClientID();

        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createAdGroupAndAd() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        jooqDbSteps.adGroupsSteps().setType(adGroupId, PhrasesAdgroupType.mobile_content);

        MobileAppAdFeatureItemMap[] features = Stream.of(MobileAppFeatureEnum.values())
                .map(feature -> new MobileAppAdFeatureItemMap().withFeature(feature).withEnabled(YesNoEnum.YES))
                .toArray(MobileAppAdFeatureItemMap[]::new);
        adId = api.userSteps.adsSteps().addAd(new AdAddItemMap()
                .withAdGroupId(adGroupId)
                .withMobileAppAd(new MobileAppAdAddMap()
                        .defaultMobileAppAd()
                        .withFeatures(features)));
    }


    @Test
    @Description("Получить Features, когда установлено ненулевое время модификации и icon_hash = NULL")
    public void getFeaturesWithNonZeroModifyTimeAndNullIconHash() {
        MobileContentRecord mobileContent = MobileContentUtils.getDefaultMobileContent(clientId);
        mobileContent.setIconHash(null);
        long mobileContentId = jooqDbSteps.mobileContentSteps().saveMobileContent(mobileContent);
        AdgroupsMobileContentRecord adgroupsMobileContentRecord =
                AdgroupsMobileContentUtils.getDefaultAdgroupsMobileContent(adGroupId, mobileContentId);
        jooqDbSteps.mobileContentSteps().saveAdgroupsMobileContent(adgroupsMobileContentRecord);

        List<AdGetItem> ads = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(adId))
                .withFieldNames(AdFieldEnum.ID)
                .withMobileAppAdFieldNames(MobileAppAdFieldEnum.FEATURES)).getAds();
        assumeThat("вернулось одно объявление", ads, hasSize(1));

        List<MobileAppAdFeatureGetItem> features = ads.get(0).getMobileAppAd().getFeatures();
        features.sort(comparator);

        List<Object> expectedFeatures = Arrays.asList(
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.YES)
                        .withFeature(MobileAppFeatureEnum.CUSTOMER_RATING).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.NO)
                        .withFeature(MobileAppFeatureEnum.ICON).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.YES)
                        .withFeature(MobileAppFeatureEnum.PRICE).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.YES)
                        .withFeature(MobileAppFeatureEnum.RATINGS).getBean());

        assertThat("только ICON является недоступной", features, beanEquivalent(expectedFeatures));
    }

    @Test
    @Description("Получить Features, когда установлено ненулевое время модификации и price_json = NULL")
    public void getFeaturesWithNonZeroModifyTimeAndNullPriceJson() {
        MobileContentRecord mobileContent = MobileContentUtils.getDefaultMobileContent(clientId);
        mobileContent.setPricesJson(null);
        long mobileContentId = jooqDbSteps.mobileContentSteps().saveMobileContent(mobileContent);
        AdgroupsMobileContentRecord adgroupsMobileContentRecord =
                AdgroupsMobileContentUtils.getDefaultAdgroupsMobileContent(adGroupId, mobileContentId);
        jooqDbSteps.mobileContentSteps().saveAdgroupsMobileContent(adgroupsMobileContentRecord);

        List<AdGetItem> ads = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(adId))
                .withFieldNames(AdFieldEnum.ID)
                .withMobileAppAdFieldNames(MobileAppAdFieldEnum.FEATURES)).getAds();
        assumeThat("вернулось одно объявление", ads, hasSize(1));

        List<MobileAppAdFeatureGetItem> features = ads.get(0).getMobileAppAd().getFeatures();
        features.sort(comparator);

        List<Object> expectedFeatures = Arrays.asList(
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.YES)
                        .withFeature(MobileAppFeatureEnum.CUSTOMER_RATING).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.YES)
                        .withFeature(MobileAppFeatureEnum.ICON).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.NO)
                        .withFeature(MobileAppFeatureEnum.PRICE).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.YES)
                        .withFeature(MobileAppFeatureEnum.RATINGS).getBean());

        assertThat("только PRICE является недоступной", features, beanEquivalent(expectedFeatures));
    }

    @Test
    @Description("Получить Features, когда установлено ненулевое время модификации и rating = NULL")
    public void getFeaturesWithNonZeroModifyTimeAndNullRating() {
        MobileContentRecord mobileContent = MobileContentUtils.getDefaultMobileContent(clientId);
        mobileContent.setRating(null);
        long mobileContentId = jooqDbSteps.mobileContentSteps().saveMobileContent(mobileContent);
        AdgroupsMobileContentRecord adgroupsMobileContentRecord =
                AdgroupsMobileContentUtils.getDefaultAdgroupsMobileContent(adGroupId, mobileContentId);
        jooqDbSteps.mobileContentSteps().saveAdgroupsMobileContent(adgroupsMobileContentRecord);

        List<AdGetItem> ads = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(adId))
                .withFieldNames(AdFieldEnum.ID)
                .withMobileAppAdFieldNames(MobileAppAdFieldEnum.FEATURES)).getAds();
        assumeThat("вернулось одно объявление", ads, hasSize(1));

        List<MobileAppAdFeatureGetItem> features = ads.get(0).getMobileAppAd().getFeatures();
        features.sort(comparator);

        List<Object> expectedFeatures = Arrays.asList(
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.NO)
                        .withFeature(MobileAppFeatureEnum.CUSTOMER_RATING).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.YES)
                        .withFeature(MobileAppFeatureEnum.ICON).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.YES)
                        .withFeature(MobileAppFeatureEnum.PRICE).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.YES)
                        .withFeature(MobileAppFeatureEnum.RATINGS).getBean());

        assertThat("только CUSTOMER_RATING является недоступной", features, beanEquivalent(expectedFeatures));
    }

    @Test
    @Description("Получить Features, когда установлено ненулевое время модификации и rating_votes = NULL")
    public void getFeaturesWithNonZeroModifyTimeAndNullRatingVotes() {
        MobileContentRecord mobileContent = MobileContentUtils.getDefaultMobileContent(clientId);
        mobileContent.setRatingVotes(null);
        long mobileContentId = jooqDbSteps.mobileContentSteps().saveMobileContent(mobileContent);
        AdgroupsMobileContentRecord adgroupsMobileContentRecord =
                AdgroupsMobileContentUtils.getDefaultAdgroupsMobileContent(adGroupId, mobileContentId);
        jooqDbSteps.mobileContentSteps().saveAdgroupsMobileContent(adgroupsMobileContentRecord);

        List<AdGetItem> ads = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(adId))
                .withFieldNames(AdFieldEnum.ID)
                .withMobileAppAdFieldNames(MobileAppAdFieldEnum.FEATURES)).getAds();
        assumeThat("вернулось одно объявление", ads, hasSize(1));

        List<MobileAppAdFeatureGetItem> features = ads.get(0).getMobileAppAd().getFeatures();
        features.sort(comparator);

        List<Object> expectedFeatures = Arrays.asList(
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.YES)
                        .withFeature(MobileAppFeatureEnum.CUSTOMER_RATING).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.YES)
                        .withFeature(MobileAppFeatureEnum.ICON).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.YES)
                        .withFeature(MobileAppFeatureEnum.PRICE).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.NO)
                        .withFeature(MobileAppFeatureEnum.RATINGS).getBean());

        assertThat("только RATINGS является недоступной", features, beanEquivalent(expectedFeatures));
    }

    @Test
    @Description("Получить Features, когда установлено ненулевое время модификации" +
            " и rating_votes = NULL и rating = NULL")
    public void getFeaturesWithNonZeroModifyTimeAndNullRatingAndRatingVotes() {
        MobileContentRecord mobileContent = MobileContentUtils.getDefaultMobileContent(clientId);
        mobileContent.setRatingVotes(null);
        mobileContent.setRating(null);
        long mobileContentId = jooqDbSteps.mobileContentSteps().saveMobileContent(mobileContent);
        AdgroupsMobileContentRecord adgroupsMobileContentRecord =
                AdgroupsMobileContentUtils.getDefaultAdgroupsMobileContent(adGroupId, mobileContentId);
        jooqDbSteps.mobileContentSteps().saveAdgroupsMobileContent(adgroupsMobileContentRecord);

        List<AdGetItem> ads = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(adId))
                .withFieldNames(AdFieldEnum.ID)
                .withMobileAppAdFieldNames(MobileAppAdFieldEnum.FEATURES)).getAds();
        assumeThat("вернулось одно объявление", ads, hasSize(1));

        List<MobileAppAdFeatureGetItem> features = ads.get(0).getMobileAppAd().getFeatures();
        features.sort(comparator);

        List<Object> expectedFeatures = Arrays.asList(
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.NO)
                        .withFeature(MobileAppFeatureEnum.CUSTOMER_RATING).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.YES)
                        .withFeature(MobileAppFeatureEnum.ICON).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.YES)
                        .withFeature(MobileAppFeatureEnum.PRICE).getBean(),
                new MobileAppAdFeatureGetItemMap()
                        .withIsAvailable(YesNoUnknownEnum.NO)
                        .withFeature(MobileAppFeatureEnum.RATINGS).getBean());

        assertThat("только RATINGS и CUSTOMER_RATING являются недоступными", features,
                beanEquivalent(expectedFeatures));
    }
}
