package ru.yandex.autotests.direct.api.ads.get.mobileappad;

import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.AdGetItem;
import com.yandex.direct.api.v5.ads.GetResponse;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PhrasesAdgroupType;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.ads.AdGetItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by semkagtn on 31.08.15.
 * https://st.yandex-team.ru/TESTIRT-6853
 */
@Aqua.Test
@Features(AdsFeatures.GET)
@Description("Проверка метода Get при получение объявлений, имеющих тип объявления с рекламой мобильного приложения")
@Issue("https://st.yandex-team.ru/DIRECT-44779")
@Tag(TagDictionary.TRUNK)
public class GetMobileAppAdTest {

    private static final String LOGIN = AdsLogins.CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);
    private static DirectJooqDbSteps jooqDbSteps;

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long campaignId;
    private static long textAdGroupId;
    private static long textAdId;

    private static Long mobileContentCampaignId;
    private static long mobileContentAdGroupId;
    private static long mobileAppAdId1;
    private static long mobileAppAdId2;

    private static Long anotherMobileContentCampaignId;
    private static long anotherMobileContentAdGroupId;
    private static long anotherMobileAppAdId;


    @BeforeClass
    public static void prepareData() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);

        textAdGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        textAdId = api.userSteps.adsSteps().addDefaultTextAd(textAdGroupId);

        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN);

        mobileContentCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        mobileContentAdGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(mobileContentCampaignId);
        jooqDbSteps.adGroupsSteps().setType(mobileContentAdGroupId, PhrasesAdgroupType.mobile_content);

        List<Long> mobileAppAdIds = api.userSteps.adsSteps().addDefaultMobileAppAds(mobileContentAdGroupId, 2);
        mobileAppAdId1 = mobileAppAdIds.get(0);
        mobileAppAdId2 = mobileAppAdIds.get(1);

        anotherMobileContentCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        anotherMobileContentAdGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(anotherMobileContentCampaignId);
        jooqDbSteps.adGroupsSteps().setType(anotherMobileContentAdGroupId, PhrasesAdgroupType.mobile_content);

        anotherMobileAppAdId = api.userSteps.adsSteps().addDefaultMobileAppAd(anotherMobileContentAdGroupId);
    }

    @Before
    public void addUnitsToClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    @Description("Получить объявление с рекламой мобильного приложения по идентификатору")
    public void getOneMobileAppAdById() {
        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withFieldNames(AdFieldEnum.ID)
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(mobileAppAdId1)));

        assertThat("вернулось одно объявление", response.getAds(), beanEquivalent(Arrays.asList(
                new AdGetItemMap()
                        .withId(mobileAppAdId1).getBean())));
    }

    @Test
    @Description("Получить два объявления с рекламой мобильного приложения по идентификаторам")
    public void getTwoMobileAppAdById() {
        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withFieldNames(AdFieldEnum.ID)
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(mobileAppAdId1, mobileAppAdId2)));

        List<AdGetItem> actualResult = response.getAds();
        assumeThat("ответ метода Get был получен", actualResult, notNullValue());

        actualResult.sort(Comparator.comparing(AdGetItem::getId));

        assertThat("вернулось два объявления", actualResult, beanEquivalent(Arrays.asList(
                new AdGetItemMap()
                        .withId(mobileAppAdId1).getBean(),
                new AdGetItemMap()
                        .withId(mobileAppAdId2).getBean())));
    }

    @Test
    @Description("Получить объявление с рекламой мобильного приложения и текстовое объявление по идентификаторам")
    public void getMobileAppAdAndTextAdById() {
        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withFieldNames(AdFieldEnum.ID)
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(mobileAppAdId1, textAdId)));

        List<AdGetItem> actualResult = response.getAds();
        assumeThat("ответ метода Get был получен", actualResult, notNullValue());

        actualResult.sort(Comparator.comparing(AdGetItem::getId));

        assertThat("вернулось два объявления", actualResult, beanEquivalent(Arrays.asList(
                new AdGetItemMap()
                        .withId(textAdId).getBean(),
                new AdGetItemMap()
                        .withId(mobileAppAdId1).getBean())));
    }

    @Test
    @Description("Получить объявление с рекламой мобильного приложений по идентификатору группы")
    public void getMobileAppAdsByAdGroupId() {
        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withFieldNames(AdFieldEnum.ID)
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withAdGroupIds(mobileContentAdGroupId)));

        List<AdGetItem> actualResult = response.getAds();
        assumeThat("ответ метода Get был получен", actualResult, notNullValue());

        actualResult.sort(Comparator.comparing(AdGetItem::getId));

        assertThat("вернулось два объявления", actualResult, beanEquivalent(Arrays.asList(
                new AdGetItemMap()
                        .withId(mobileAppAdId1).getBean(),
                new AdGetItemMap()
                        .withId(mobileAppAdId2).getBean())));
    }

    @Test
    @Description("Получить объявления с рекламой мобильных приложений по идентификаторам двух групп")
    public void getMobileAppAdsByAdGroupIds() {
        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withFieldNames(AdFieldEnum.ID)
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withAdGroupIds(mobileContentAdGroupId, anotherMobileContentAdGroupId)));

        List<AdGetItem> actualResult = response.getAds();
        assumeThat("ответ метода Get был получен", actualResult, notNullValue());

        actualResult.sort(Comparator.comparing(AdGetItem::getId));

        assertThat("вернулось три объявления", actualResult, beanEquivalent(Arrays.asList(
                new AdGetItemMap()
                        .withId(mobileAppAdId1).getBean(),
                new AdGetItemMap()
                        .withId(mobileAppAdId2).getBean(),
                new AdGetItemMap()
                        .withId(anotherMobileAppAdId).getBean())));
    }

    @Test
    @Description("Получить объявление с рекламой мобильных приложений и текстовое объявление по идентефикаторам групп")
    public void getMobileAppAdAndTextAdByAdGroupIds() {
        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withFieldNames(AdFieldEnum.ID)
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withAdGroupIds(anotherMobileContentAdGroupId, textAdGroupId)));

        List<AdGetItem> actualResult = response.getAds();
        assumeThat("ответ метода Get был получен", actualResult, notNullValue());

        actualResult.sort(Comparator.comparing(AdGetItem::getId));

        assertThat("вернулось два объявления", actualResult, beanEquivalent(Arrays.asList(
                new AdGetItemMap()
                        .withId(textAdId).getBean(),
                new AdGetItemMap()
                        .withId(anotherMobileAppAdId).getBean())));
    }

    @Test
    @Description("Получить объявления с рекламой мобильных приложений и" +
            " текстовые объявления по идентификаторам кампании")
    public void getMobileAppAdsAndTextAdsByCampaignId() {
        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withFieldNames(AdFieldEnum.ID)
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId, mobileContentCampaignId)));

        List<AdGetItem> actualResult = response.getAds();
        assumeThat("ответ метода Get был получен", actualResult, notNullValue());

        actualResult.sort(Comparator.comparing(AdGetItem::getId));

        assertThat("вернулось четыре объявления", actualResult, beanEquivalent(Arrays.asList(
                new AdGetItemMap()
                        .withId(textAdId).getBean(),
                new AdGetItemMap()
                        .withId(mobileAppAdId1).getBean(),
                new AdGetItemMap()
                        .withId(mobileAppAdId2).getBean())));
    }

    @Test
    @Description("Получить объявления с рекламой мобильных приложений и" +
            " текстовые объявления по двум идентификаторам кампаний")
    public void getMobileAppAdsAndTextAdsByTwoCampaignIds() {
        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withFieldNames(AdFieldEnum.ID)
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withCampaignIds(campaignId, anotherMobileContentCampaignId, mobileContentCampaignId)));

        List<AdGetItem> actualResult = response.getAds();
        assumeThat("ответ метода Get был получен", actualResult, notNullValue());

        actualResult.sort(Comparator.comparing(AdGetItem::getId));

        assertThat("вернулось четыре объявления", actualResult, beanEquivalent(Arrays.asList(
                new AdGetItemMap()
                        .withId(textAdId).getBean(),
                new AdGetItemMap()
                        .withId(mobileAppAdId1).getBean(),
                new AdGetItemMap()
                        .withId(mobileAppAdId2).getBean(),
                new AdGetItemMap()
                        .withId(anotherMobileAppAdId).getBean())));
    }
}
