package ru.yandex.autotests.direct.api.ads.suspend;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.ads.SuspendRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by pavryabov on 16.04.15.
 * https://st.yandex-team.ru/TESTIRT-4842
 */
@Aqua.Test
@Features(AdsFeatures.SUSPEND)
@Description("Проверка правильности списания баллов")
@Issue("https://st.yandex-team.ru/DIRECT-34573")
public class AdsSuspendUnitsTest {
    private static final String CLIENT = AdsLogins.SUSPEND_UNITS;

    private static final int CALL_COST = 15;
    private static final int AD_COST = 0;
    private static final int DEFAULT_ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR = 50;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().version(104).as(CLIENT);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long adId1;
    private Long adId2;

    @Before
    public void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(CLIENT);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        adId1 = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        adId2 = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        api.userSteps.bannersFakeSteps().makeBannersModerated(adId1, adId2);
    }

    @Test
    public void suspendOneAd() {
        //DIRECT-41183
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adsSteps().adsSuspend(adId1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + AD_COST));
    }

    @Test
    public void suspendTwoAds() {
        //DIRECT-41183
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adsSteps().adsSuspend(adId1, adId2);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + 2 * AD_COST));
    }

    @Test
    public void suspendOneAdWithError() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adsSteps().adsSuspend(
                new SuspendRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(-adId1)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + DEFAULT_ERROR_COST));
    }

    @Test
    public void suspendTwoAdsWithOneError() {
        //DIRECT-41183
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adsSteps().adsSuspend(
                new SuspendRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(adId1, -adId1)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + DEFAULT_ERROR_COST + AD_COST));
    }

    @Test
    public void suspendTwoAdsWithTwoErrors() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adsSteps().adsSuspend(
                new SuspendRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(-adId1, -adId1)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + 2 * DEFAULT_ERROR_COST));
    }

    @Test
    public void suspendWithInvalidRequest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adsSteps().expectErrorOnAdsSuspend(
                new SuspendRequestMap().withSelectionCriteria(new IdsCriteriaMap()),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        path(IdsCriteriaMap.IDS)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(COMMON_REQUEST_ERROR));
    }

    @Test
    public void callAdsSuspendWithZeroUnits() {
        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(CLIENT, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assumeThat("у клиента не осталось баллов", units, equalTo(0));
        api.userSteps.adsSteps().expectErrorOnAdsSuspend(new SuspendRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap().withIds(adId1)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION)
        );
    }
}
