
package ru.yandex.autotests.direct.api.ads.turbopages;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.turbopages.GetResponse;
import com.yandex.direct.api.v5.turbopages.TurboPageFieldEnum;
import com.yandex.direct.api.v5.turbopages.TurboPageGetItem;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.api.ads.TurboPagesFeatures;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.TurbolandingsRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.general.LimitOffsetMap;
import ru.yandex.autotests.directapi.model.api5.turbopages.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.turbopages.TurboPageGetItemMap;
import ru.yandex.autotests.directapi.model.api5.turbopages.TurboPagesCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.TestSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.on;
import static ch.lambdaj.Lambda.sort;
import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasItems;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by adubinkin on 18/04/19.
 * https://st.yandex-team.ru/DIRECT-89837
*/

@Aqua.Test
@Features(TurboPagesFeatures.GET)
@Description("Проверка получения турбо-страниц клиента")
@RunWith(Parameterized.class)
public class GetTest {

    private static String client = AdsLogins.CLIENT;
    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public ProtocolType protocol;

    private GetRequestMap requestMap;

    private static long firstId;
    private static long secondId;
    private static long thirdId;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        return  Arrays.asList(new Object[][]{
                {ProtocolType.JSON},
                {ProtocolType.SOAP}
        });
    }

    @BeforeClass
    public static void createTurboPages() {
        DirectJooqDbSteps jooqSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(client);
        long clientId = jooqSteps.usersSteps().getUser(client).getClientid();

        firstId  = jooqSteps.turboLandingsSteps().createDefaultTurbolanding(clientId);
        secondId = jooqSteps.turboLandingsSteps().createDefaultTurbolanding(clientId);
        thirdId  = jooqSteps.turboLandingsSteps().createDefaultTurbolanding(clientId);
    }

    @AfterClass
    public static void deleteTurboPages() {
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(client).turboLandingsSteps()
                .deleteTurboLandings(asList(firstId, secondId, thirdId));
    }

    @Before
    public void initRequestMap() {
        api.protocol(protocol);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        requestMap = new GetRequestMap()
                .withSelectionCriteria(new TurboPagesCriteriaMap()
                        .withIds(firstId, secondId, thirdId))
                .withFieldNames(TurboPageFieldEnum.ID);
    }

    @Test
    @Description("Получение турбо-страниц без использования дополнительных параметров")
    public void getTurboPages() {
        List<TurboPageGetItem> actualResult = api.userSteps.turboPagesSteps().get(requestMap);
        actualResult = sort(actualResult, on(TurboPageGetItem.class).getId());

        assertThat("вернулись предполагаемые турбо-страницы", actualResult, beanEquivalent(asList(
                (TurboPageGetItem) new TurboPageGetItemMap()
                        .withId(firstId).getBean(),
                (TurboPageGetItem) new TurboPageGetItemMap()
                        .withId(secondId).getBean(),
                (TurboPageGetItem) new TurboPageGetItemMap()
                        .withId(thirdId).getBean())));
    }

    @Test
    @Description("Получение турбо-страницы c полным набором полей")
    public void getTurboPagesWithAllFields() {
        requestMap
                .withFieldNames(TurboPageFieldEnum.values())
                .withSelectionCriteria(new TurboPagesCriteriaMap()
                        .withIds(firstId));
        TurbolandingsRecord expectedTurboLanding =
                api.userSteps.getDirectJooqDbSteps().turboLandingsSteps().getTurboLanding(firstId);
        List<TurboPageGetItem> actualResult = api.userSteps.turboPagesSteps().get(requestMap);

        assertThat("вернулись все запрошенные поля", actualResult, beanEquivalent(asList(
                (TurboPageGetItem) new TurboPageGetItemMap()
                        .withId(firstId)
                        .withHref(expectedTurboLanding.getHref())
                        .withName(expectedTurboLanding.getName())
                        .withPreviewHref(expectedTurboLanding.getPreviewHref())
                        .getBean())));
    }

    @Test
    @Description("Получение всех турболендингов клиента, если в запросе не задан SelectionCriteria")
    public void getAllTurboPagesForClient() {
        requestMap.withSelectionCriteria(null);
        List<TurboPageGetItem> actualResult = api.userSteps.turboPagesSteps().get(requestMap);
        List<Long> actualTurboPageIds = actualResult.stream().map(TurboPageGetItem::getId)
                .collect(Collectors.toList());

        assertThat("вернулись все турбо-страницы клиента",
                actualTurboPageIds, hasItems(firstId, secondId, thirdId));
    }

    @Test
    @Description("Получение турбо-страниц с использованием лимита")
    public void getTurboPagesUsingLimit() {
        List<TurboPageGetItem> resultWithoutLimit = api.userSteps.turboPagesSteps().get(requestMap);
        assumeThat("вернулись три турбо-страницы", resultWithoutLimit, hasSize(3));

        List<TurboPageGetItem> actualResult = api.userSteps.turboPagesSteps().get(requestMap
                .withPage(new LimitOffsetMap()
                        .withLimit(2L)));

        assertThat("вернулись первые две турбо-страницы", actualResult, beanEquivalent(asList(
                resultWithoutLimit.get(0), resultWithoutLimit.get(1))));
    }

    @Test
    @Description("Получение турбо-страниц с использованием смещения")
    public void getTurboPagesUsingOffset() {
        List<TurboPageGetItem> resultWithoutOffset = api.userSteps.turboPagesSteps().get(requestMap);
        assumeThat("вернулись три турбо-страницы", resultWithoutOffset, hasSize(3));

        List<TurboPageGetItem> actualResult = api.userSteps.turboPagesSteps().get(requestMap
                .withPage(new LimitOffsetMap()
                        .withOffset(1L)));

        assertThat("вернулись турбо-страницы начиная со второй", actualResult, beanEquivalent(asList(
                resultWithoutOffset.get(1), resultWithoutOffset.get(2))));
    }

    @Test
    @Description("Получение турбо-страниц с использованием лимита и смещения")
    public void getTurboPagesUsingLimitAndOffset() {
        List<TurboPageGetItem> resultWithoutOffset = api.userSteps.turboPagesSteps().get(requestMap);
        assumeThat("вернулись три турбо-страницы", resultWithoutOffset, hasSize(3));

        List<TurboPageGetItem> actualResult = api.userSteps.turboPagesSteps().get(requestMap
                .withPage(new LimitOffsetMap()
                        .withLimit(1L)
                        .withOffset(1L)));

        assertThat("вернулась вторая турбо-страница", actualResult, beanEquivalent(singletonList(
                resultWithoutOffset.get(1))));
    }

    @Test
    @Description("Проверка возвращаемого значения LimitedBy при заданном лимите")
    public void checkLimitedByIfLimitIsWorked() {
        GetResponse response = api.userSteps.turboPagesSteps().turboPagesGet(requestMap
                .withPage(new LimitOffsetMap()
                        .withLimit(1L)));
        TestSteps.assertThat("значение LimitedBy верное", response.getLimitedBy(), equalTo(1L));
    }

    @Test
    @Description("Проверка возвращаемого значения LimitedBy при заданном лимите и смещении")
    public void checkLimitedByIfLimitIsWorkedWithOffset() {
        GetResponse response = api.userSteps.turboPagesSteps().turboPagesGet(requestMap
                .withPage(new LimitOffsetMap()
                        .withLimit(1L)
                        .withOffset(1L)));
        TestSteps.assertThat("значение LimitedBy верное", response.getLimitedBy(), equalTo(2L));
    }

    @Test
    @Description("Получение турбо-страниц по 'красивому' url")
    @Ignore
    /**
     * Тест нужно запускать вручную, т.к. искомая страница должна быть не только в БД, но и в конструкторе,
     * что через степы сделать нельзя. Если база переналилась и такой страницы нет - нужно создать ее
     * через конструктор турболендингов на ТС для клиента ads-client и нажать "Опубликовать"
     */
    public void getTurboPagesWithBoundWithHrefs() {
        requestMap
                .withFieldNames(TurboPageFieldEnum.values())
                .withSelectionCriteria(new TurboPagesCriteriaMap()
                        .withBoundWithHrefs("https://test-page.turbo-site-test.common.yandex.net"));

        List<TurboPageGetItem> actualResult = api.userSteps.turboPagesSteps().get(requestMap);

        assertThat("нашлась одна турбо-=страница", actualResult, hasSize(1));
    }

}
