package ru.yandex.autotests.direct.api.ads.unarchive;

import java.util.List;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.AdGetItem;
import com.yandex.direct.api.v5.ads.GetResponse;
import com.yandex.direct.api.v5.general.StateEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.ArchiveRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.UnarchiveRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.stream.Collectors.toList;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 16.04.15.
 * https://st.yandex-team.ru/TESTIRT-5211
 */
@Aqua.Test
@Features(AdsFeatures.UNARCHIVE)
@Description("Проверка правильности списания баллов при разархивировании объявления")
@Issue("https://st.yandex-team.ru/DIRECT-34573")
public class AdsUnarchiveUnitsTest {
    private static final String CLIENT = AdsLogins.UNARCHIVE_UNITS;

    private static final int CALL_COST = 40;
    private static final int AD_COST = 0;
    private static final int DEFAULT_ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR = 50;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().version(104).as(CLIENT);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long adId1;
    private Long adId2;

    @Before
    @Step("Подготовка данных для теста")
    public void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(CLIENT);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        List<Long> adIds = api.userSteps.adsSteps().addDefaultTextAds(adGroupId, 2);
        adId1 = adIds.get(0);
        adId2 = adIds.get(1);

        api.userSteps.bannersFakeSteps().makeBannersModerated(adId1, adId2);
        api.userSteps.adsSteps().adsArchive(adId1, adId2);

        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withIds(adId1, adId2))
                .withFieldNames(AdFieldEnum.ID, AdFieldEnum.STATE));
        List<StateEnum> states = response.getAds().stream().map(AdGetItem::getState).collect(toList());
        assumeThat("объявления заархивированы", states, contains(StateEnum.ARCHIVED, StateEnum.ARCHIVED));
    }

    @Test
    @Description("Разархивировать одно объявление")
    public void unarchiveOneAd() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adsSteps().adsUnarchive(adId1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + AD_COST));
    }

    @Test
    @Description("Разархивировать два объявления")
    public void unarchiveTwoAds() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adsSteps().adsUnarchive(adId1, adId2);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + 2 * AD_COST));
    }

    @Test
    @Description("Запрос на разархивирование объявления с ошибкой")
    public void unarchiveOneAdWithError() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adsSteps().adsUnarchive(-adId1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + DEFAULT_ERROR_COST));
    }

    @Test
    @Description("Запрос на разархивирование двух объявлений с одной ошибкой")
    public void unarchiveTwoAdsWithOneError() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adsSteps().adsUnarchive(adId1, -adId1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + DEFAULT_ERROR_COST + AD_COST));
    }

    @Test
    @Description("Запрос на разархивирование объявлений с двумя ошибками")
    public void unarchiveTwoAdsWithTwoErrors() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adsSteps().adsUnarchive(-adId1, -adId1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + 2 * DEFAULT_ERROR_COST));
    }

    @Test
    @Description("Невалидный запрос на разархивацию объявления")
    public void unarchiveWithInvalidRequest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adsSteps().expectErrorOnAdsUnarchive(
                new UnarchiveRequestMap().withSelectionCriteria(new IdsCriteriaMap()),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        path(IdsCriteriaMap.IDS)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(COMMON_REQUEST_ERROR));
    }

    @Test
    @Description("Запрос на разархивацию объявления, когда у клиента отсутствуют баллы")
    public void callAdsunarchiveWithZeroUnits() {
        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(CLIENT, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assumeThat("у клиента не осталось баллов", units, equalTo(0));
        api.userSteps.adsSteps().expectErrorOnAdsArchive(new ArchiveRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap().withIds(adId1)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION)
        );
    }
}
