package ru.yandex.autotests.direct.api.ads.unarchive;

import java.util.Collection;
import java.util.List;
import java.util.stream.Stream;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.AdGetItem;
import com.yandex.direct.api.v5.ads.GetResponse;
import com.yandex.direct.api.v5.general.StateEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.utils.model.ShardNumbers;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.Matchers.contains;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 16.04.15.
 * https://st.yandex-team.ru/TESTIRT-5211
 */
@Aqua.Test
@Features(AdsFeatures.UNARCHIVE)
@Description("Проверка прав доступа агентства к объявлению при его разархивации")
@Issue("https://st.yandex-team.ru/DIRECT-34573")
@RunWith(Parameterized.class)
public class AgencyCheckRightsToIdTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static String client = AdsLogins.CLIENT;
    private static String agency = AdsLogins.AGENCY;
    private static String agencyRep = AdsLogins.AGENCY_REP;
    private static String agencyRepCl = AdsLogins.AGENCY_REP_CL;
    private static String subclient = AdsLogins.SUBCLIENT;
    private static String subclientElse = AdsLogins.SUBCLIENT_ELSE;

    private static Long clientAdId;
    private static Long subclientAdId;
    private static Long subclientElseAdId;

    @Parameterized.Parameter(0)
    public String loginForAuthorization;

    @Parameterized.Parameters(name = "loginForAuthorization = {0}")
    public static Collection testData() {
        Object[][] data = new Object[][]{
                {agency},
                {agencyRep},
                {agencyRepCl}
        };
        return asList(data);
    }

    @BeforeClass
    public static void createNotMineCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(agency);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(subclient);
        api.userSteps.clientFakeSteps().reshardUserWithLock(subclient, ShardNumbers.DEFAULT_SHARD);
        api.userSteps.clientFakeSteps().reshardUserWithLock(subclientElse, ShardNumbers.DEFAULT_SHARD);
        api.as(client);
        Long clientCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        Long clientAdGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(clientCampaignId);
        api.userSteps.keywordsSteps().addDefaultKeyword(clientAdGroupId);
        clientAdId = api.userSteps.adsSteps().addDefaultTextAd(clientAdGroupId);
        api.as(agency);
        Long subclientCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(subclient);
        Long subclientAdGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(subclientCampaignId, subclient);
        api.userSteps.keywordsSteps().addDefaultKeyword(subclient, subclientAdGroupId);
        subclientAdId = api.userSteps.adsSteps().addDefaultTextAd(subclientAdGroupId, subclient);
        Long subclientElseCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(subclientElse);
        Long subclientElseAdGroupId =
                api.userSteps.adGroupsSteps().addDefaultGroup(subclientElseCampaignId, subclientElse);
        api.userSteps.keywordsSteps().addDefaultKeyword(subclientElse, subclientElseAdGroupId);
        subclientElseAdId = api.userSteps.adsSteps().addDefaultTextAd(subclientElseAdGroupId, subclientElse);

        api.as(Logins.SUPER_LOGIN);
        api.userSteps.bannersFakeSteps().makeBannersModerated(
                clientAdId, subclientAdId, subclientElseAdId);
    }

    @Before
    public void unarchiveAds() {
        api.as(Logins.SUPER_LOGIN);
        api.userSteps.adsSteps().adsArchive(client, clientAdId);
        api.userSteps.adsSteps().adsArchive(subclient, subclientAdId);
        api.userSteps.adsSteps().adsArchive(subclientElse, subclientElseAdId);

        GetResponse responseClient = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withIds(clientAdId))
                .withFieldNames(AdFieldEnum.ID, AdFieldEnum.STATE),
                client);
        GetResponse responseSubclient = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                        .withSelectionCriteria(new AdsSelectionCriteriaMap().withIds(subclientAdId))
                        .withFieldNames(AdFieldEnum.ID, AdFieldEnum.STATE),
                subclient);
        GetResponse responseSubclientElse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                        .withSelectionCriteria(new AdsSelectionCriteriaMap().withIds(subclientElseAdId))
                        .withFieldNames(AdFieldEnum.ID, AdFieldEnum.STATE),
                subclientElse);
        List<StateEnum> states = Stream.of(responseClient.getAds(), responseSubclient.getAds(), responseSubclientElse.getAds())
                .flatMap(ads -> ads.stream().map(AdGetItem::getState))
                .collect(toList());
        assumeThat("объявления заархивированы", states,
                contains(StateEnum.ARCHIVED, StateEnum.ARCHIVED, StateEnum.ARCHIVED));
        api.as(loginForAuthorization);
    }

    @Test
    public void callUnarchiveWithSubclientId() {
        api.userSteps.bannersFakeSteps().makeBannersModerated(subclientAdId);
        api.userSteps.adsSteps().shouldGetResultOnUnarchive(
                new IdsCriteriaMap().withIds(subclientAdId),
                subclient,
                ExpectedResult.success()
        );
    }

    @Test
    public void callUnarchiveWithSubclientElseId() {
        api.userSteps.adsSteps().shouldGetResultOnUnarchive(
                new IdsCriteriaMap().withIds(subclientElseAdId),
                subclient,
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AD_NOT_FOUND))
        );
    }

    @Test
    public void callUnarchiveWithIdsOfBothSubclients() {
        api.userSteps.bannersFakeSteps().makeBannersModerated(subclientAdId);
        api.userSteps.adsSteps().shouldGetResultOnUnarchive(
                new IdsCriteriaMap().withIds(subclientAdId, subclientElseAdId),
                subclient,
                ExpectedResult.success(),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AD_NOT_FOUND))
        );
    }

    @Test
    public void callUnarchiveWithClientId() {
        api.userSteps.adsSteps().shouldGetResultOnUnarchive(
                new IdsCriteriaMap().withIds(clientAdId),
                subclient,
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AD_NOT_FOUND))
        );
    }
}
