package ru.yandex.autotests.direct.api.ads.unarchive;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.GetResponse;
import com.yandex.direct.api.v5.general.StateEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Collections.singletonList;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by semkagtn on 16.04.15.
 * https://st.yandex-team.ru/TESTIRT-5211
 */
@Aqua.Test
@Features(AdsFeatures.UNARCHIVE)
@Description("Пообъектная обработка запроса разархивирования объявлений")
@Issue("https://st.yandex-team.ru/DIRECT-34573")
@Tag(TagDictionary.TRUNK)
public class DiscreteValidationTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(AdsLogins.CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long adId;

    @Before
    @Step("Подготовка данных для теста")
    public void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdsLogins.CLIENT);
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(AdsLogins.CLIENT);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        adId = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        api.userSteps.bannersFakeSteps().makeBannersModerated(adId);
        api.userSteps.adsSteps().adsArchive(adId);

        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withIds(adId))
                .withFieldNames(AdFieldEnum.ID, AdFieldEnum.STATE));
        StateEnum state = response.getAds().get(0).getState();
        assertThat("объявление в архиве", state, equalTo(StateEnum.ARCHIVED));
    }

    @Test
    @Description("Разархивировать валидное и невалидное объявления")
    public void unarchiveValidAndInvalidAds() {
        api.userSteps.adsSteps()
                .shouldGetResultOnUnarchive(new IdsCriteriaMap().withIds(adId, -adId),
                        ExpectedResult.success(),
                        ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AD_NOT_FOUND)));

        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withIds(adId))
                .withFieldNames(AdFieldEnum.ID, AdFieldEnum.STATE));
        StateEnum state = response.getAds().get(0).getState();
        assertThat("объявление разархивировалось", state, not(equalTo(StateEnum.ARCHIVED)));
    }

    @Test
    @Description("Разархивировать невалидное и валидное объявления")
    public void unarchiveInvalidAndValidAds() {
        api.userSteps.adsSteps()
                .shouldGetResultOnUnarchive(new IdsCriteriaMap().withIds(-adId, adId),
                        ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.AD_NOT_FOUND)),
                        ExpectedResult.success());

        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withIds(adId))
                .withFieldNames(AdFieldEnum.ID, AdFieldEnum.STATE));
        StateEnum state = response.getAds().get(0).getState();
        assertThat("объявление разархивировалось", state, not(equalTo(StateEnum.ARCHIVED)));
    }

    @Test
    @Description("Разархивировать два невалидных объявления")
    public void unarchiveTwoInvalidAds() {
        Notification duplicateWarning = new Notification(10000, Api5ErrorDetails.AD_MORE_THAN_ONCE_IN_REQUEST);
        Notification notFoundError = new Notification(8800, Api5ErrorDetails.AD_NOT_FOUND);
        JavaOrPerlExpectedResult expectedResult = JavaOrPerlExpectedResult
                .java(ExpectedResult.create(nullValue(), singletonList(duplicateWarning),
                        singletonList(notFoundError)))
                .perl(ExpectedResult.errors(notFoundError));
        ;

        api.userSteps.adsSteps().shouldGetResultOnUnarchive(new IdsCriteriaMap().withIds(-adId, -adId), null,
                expectedResult, expectedResult);
    }
}
