package ru.yandex.autotests.direct.api.ads.unarchive;

import java.sql.Timestamp;
import java.util.Arrays;
import java.util.Date;

import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusarch;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusbssynced;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusshow;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersRecord;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.enums.AdGroupType;
import ru.yandex.autotests.directapi.enums.BannerType;
import ru.yandex.autotests.directapi.enums.CampaignType;
import ru.yandex.autotests.directapi.model.api5.ads.UnarchiveRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.common.Value;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 27.08.15.
 * https://st.yandex-team.ru/TESTIRT-6853
 */
@Aqua.Test
@Features(AdsFeatures.UNARCHIVE)
@Description("Проверка разархивации нетекстовых объявлений")
@Issue("https://st.yandex-team.ru/DIRECT-44779")
@RunWith(Parameterized.class)
@Tag(TagDictionary.TRUNK)
public class UnarchiveNonTextAdsTest {

    private static final String LOGIN = AdsLogins.CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static int shard;
    private long adGroupId;

    @Parameterized.Parameter(0)
    public CampaignType campaignType;

    @Parameterized.Parameter(1)
    public AdGroupType adGroupType;

    @Parameterized.Parameter(2)
    public BannerType bannerType;

    @Parameterized.Parameters(name = "bannerType = {2}")
    public static Iterable<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {CampaignType.MOBILE_CONTENT, AdGroupType.MOBILE_CONTENT, BannerType.MOBILE_CONTENT},
                {CampaignType.DYNAMIC, AdGroupType.DYNAMIC, BannerType.DYNAMIC},
        });
    }

    @BeforeClass
    public static void createAdGroup() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    private long adId;

    @Before
    @Step("Подготовка данных для теста")
    public void createAd() {
        createCampaignWithNotEmptyAdGroup();

        adId = api.userSteps.adsSteps().addDefaultAd(adGroupId, bannerType);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(adId);
        api.userSteps.adsSteps().adsArchive(adId);

        api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN)
                .bannersSteps()
                .setBannerStatusArchived(adId, BannersStatusarch.Yes);
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN)
                .bannersSteps()
                .setBannerStatusShow(adId, BannersStatusshow.No);
        BannersRecord actualBanner = api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN)
                .bannersSteps()
                .getBanner(adId);
        assumeThat("объявление является архивированным", actualBanner.getStatusarch(),
                is(BannersStatusarch.Yes));
        assumeThat("объявление не показывается", actualBanner.getStatusshow(),
                is(BannersStatusshow.No));
    }

    private void createCampaignWithNotEmptyAdGroup() {
        Long campaignId = api.userSteps.campaignSteps().addDefaultCampaign(campaignType);
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN).campaignsSteps()
                .setCampaignsStatusModerate(campaignId, CampaignsStatusmoderate.Yes);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId, adGroupType);

        if (adGroupType == AdGroupType.DYNAMIC) {
            api.userSteps.dynamicTextAdTargetsSteps().addDefaultWebpage(adGroupId);
        } else {
            api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
        }
    }

    @Test
    @Description("Проверка того, что объявление с рекламой мобильного приложения успешно разархивируется")
    public void unarchiveAdCheckIfArchived() {
        api.userSteps.adsSteps().shouldGetResultOnUnarchive(new UnarchiveRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(adId)),
                ExpectedResult.success(adId));

        BannersRecord actualBanner = api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN)
                .bannersSteps()
                .getBanner(adId);

        assertThat("объявление не является архивированным", actualBanner.getStatusarch(),
                is(BannersStatusarch.No));
        assertThat("объявление готово к модерации", actualBanner.getStatusmoderate(),
                is(BannersStatusmoderate.Ready));
        assertThat("объявление не промодерировано", actualBanner.getStatuspostmoderate(),
                is(BannersStatuspostmoderate.No));
    }

    @Test
    @Description("Проверка обновления поля LastChange при разархивации объявления с рекламой мобильного приложения")
    public void unarchiveAdCheckLastChange() {
        api.userSteps.bannersFakeSteps().setLastChange(adId, DateTime.now().minusDays(7).toString());

        api.userSteps.adsSteps().shouldGetResultOnUnarchive(new UnarchiveRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(adId)),
                ExpectedResult.success(adId));


        Timestamp actualLastChange = api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN)
                .bannersSteps()
                .getBanner(adId)
                .getLastchange();

        LocalDate actualLastChangeDate = LocalDate.fromDateFields(Date.from(actualLastChange.toInstant()));
        LocalDate expectedLastChangeDate = DateTime.now().toLocalDateTime().toLocalDate();


        assertThat("поле LastChange было обновлено", actualLastChangeDate, equalTo(expectedLastChangeDate));
    }

    @Test
    @Description("Проверка сбрасывания StatusBsSynced при разархиввации объявления с рекламой мобильного приложения")
    public void unarchiveAdCheckStatusBsSynced() {
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adId, Value.YES);

        api.userSteps.adsSteps().shouldGetResultOnUnarchive(new UnarchiveRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(adId)),
                ExpectedResult.success(adId));

        BannersStatusbssynced actualStatusBsSynced = api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN)
                .bannersSteps()
                .getBanner(adId)
                .getStatusbssynced();
        BannersStatusbssynced expectedStatusBsSynced = BannersStatusbssynced.No;

        assertThat("объявлению был сброшен StatusBsSynced", actualStatusBsSynced,
                is(expectedStatusBsSynced));
    }
}
