package ru.yandex.autotests.direct.api.ads.update;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.common.RegionIDValues;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 22.04.15.
 * https://st.yandex-team.ru/TESTIRT-5113
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Description("Изменение статусов модерации объектов при обновлении объявления")
@Tag(TagDictionary.TRUNK)
public class UpdateAdStatusesModerateTest {

    //DIRECT-49087

    private static final String LOGIN = AdsLogins.CLIENT;
    private static final String NEW_TITLE = "This is new ad title";

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).protocol(ProtocolType.JSON).as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long campaignID;
    private long adGroupID;
    private long adID;

    @Before
    @Step("Подготовка данных для теста")
    public void createAd() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        adGroupID = api.userSteps.adGroupsSteps().addGroup(new AdGroupAddItemMap()
                .defaultAdGroupAddItem(campaignID)
                .withRegionIds(Long.parseLong(RegionIDValues.ANY_REGION_ID)));
        api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);
        adID = api.userSteps.adsSteps().addDefaultTextAd(adGroupID);
    }

    @Test
    @Description("Обновить объявление-черновик")
    public void updateNewAdInNewCampaign() {
        String campaignStatusModerate = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(campaignID).getStatusModerate();
        assumeThat("кампания является черновиком", campaignStatusModerate, equalTo(Status.NEW));

        String bannerStatusModerate = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusModerate();
        assumeThat("объявление является черновиком", bannerStatusModerate, equalTo(Status.NEW));

        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adID)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd()
                                .withTitle(NEW_TITLE))));

        bannerStatusModerate = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusModerate();
        assertThat("объявление осталось черновиком", bannerStatusModerate, equalTo(Status.NEW));
    }

    @Test
    @Description("Обновить объявление-черновик в активной кампании")
    public void updateNewAdInActiveCampaign() {
        api.userSteps.campaignFakeSteps().makeCampaignActive(campaignID);

        String campaignStatusActive = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(campaignID).getStatusActive();
        assumeThat("кампания является активной", campaignStatusActive, equalTo(Status.YES));

        String bannerStatusModerate = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusModerate();
        assumeThat("объявление является черновиком", bannerStatusModerate, equalTo(Status.NEW));

        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adID)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd()
                                .withTitle(NEW_TITLE))));

        bannerStatusModerate = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusModerate();
        assertThat("объявление осталось черновиком", bannerStatusModerate, equalTo(Status.NEW));
    }

    @Test
    @Description("Обновить объявление, когда у кампании уже существует объвяление на модерации")
    public void updateAdForCampaignWithAdOnModeration() {
        long anotherAdID = api.userSteps.adsSteps().addDefaultTextAd(adGroupID);
        api.userSteps.adsSteps().adsModerate(anotherAdID);

        String anotherBannerStatusModerate = api.userSteps.bannersFakeSteps().getBannerParams(anotherAdID).getStatusModerate();
        assumeThat("объявление находится на модерации", anotherBannerStatusModerate, equalTo(Status.READY));

        String campaignStatusModerate = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(campaignID).getStatusModerate();
        assumeThat("кампания находится на модерации", campaignStatusModerate, equalTo(Status.READY));

        String bannerStatusModerate = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusModerate();
        assumeThat("объявление является черновиком", bannerStatusModerate, equalTo(Status.NEW));

        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adID)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd()
                                .withTitle(NEW_TITLE))));

        bannerStatusModerate = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusModerate();
        assertThat("объявление осталось черновиком", bannerStatusModerate, equalTo(Status.NEW));
    }

    @Test
    @Description("Обновить объявление на модерации")
    public void updateAdOnModeration() {
        api.userSteps.bannersFakeSteps().setStatusModerate(adID, Status.READY);

        String campaignStatusModerate = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(campaignID).getStatusModerate();
        assumeThat("кампания является черновиком", campaignStatusModerate, equalTo(Status.NEW));

        String bannerStatusModerate = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusModerate();
        assumeThat("объявление находится на модерации", bannerStatusModerate, equalTo(Status.READY));

        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adID)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd()
                                .withTitle(NEW_TITLE))));

        bannerStatusModerate = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusModerate();
        assertThat("объявление осталось на модерации", bannerStatusModerate, equalTo(Status.READY));
    }

    @Test
    @Description("Обновить активное объявление")
    public void updateActiveAd() {
        api.userSteps.makeCampaignActiveV5(LOGIN, campaignID);
        api.userSteps.bannersFakeSteps().makeBannerActive(adID);

        String campaignStatusActive = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(campaignID).getStatusActive();
        assumeThat("кампания является активной", campaignStatusActive, equalTo(Status.YES));

        String bannerStatusActive = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusActive();
        assumeThat("объявление является активным", bannerStatusActive, equalTo(Status.YES));

        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adID)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd()
                                .withTitle(NEW_TITLE))));

        String bannerStatusModerate = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusModerate();
        assertThat("статус модерации объявления сброшен", bannerStatusModerate, equalTo(Status.READY));
    }

    @Test
    @Description("Обновить отклонённое на модерации объявление")
    public void updateDeclinedAd() {
        api.userSteps.bannersFakeSteps().makeBannersDeclined(adID);

        api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN).campaignsSteps()
                .setCampaignsStatusModerate(campaignID, CampaignsStatusmoderate.Yes);
        String campaignStatusActive = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(campaignID).getStatusModerate();
        assumeThat("кампания не является черновиком", campaignStatusActive, not(equalTo(Status.NEW)));

        String bannerStatusModerate = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusModerate();
        assumeThat("объявление является отклонённым на модерации", bannerStatusModerate, equalTo(Status.NO));

        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adID)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd()
                                .withTitle(NEW_TITLE))));

        bannerStatusModerate = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusModerate();
        assertThat("статус модерации объявления сброшен", bannerStatusModerate, equalTo(Status.READY));
    }

    @Test
    @Description("Если баннер черновой то при update мы явно ставим ему statusPostModearte=No")
    public void updateNewAdAndCheckStatusPostModerate() {
        api.userSteps.bannersFakeSteps().setStatusPostModerate(adID, Status.YES);
        String bannerStatusModerate = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusModerate();
        String bannerStatusPostModerate = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusPostModerate();
        assumeThat("объявление является черновиком", bannerStatusModerate, equalTo(Status.NEW));
        assumeThat("StatusPostModerate = Yes", bannerStatusPostModerate, equalTo(Status.YES));

        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adID)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd()
                                .withTitle(NEW_TITLE))));

        bannerStatusPostModerate = api.userSteps.bannersFakeSteps().getBannerParams(adID).getStatusPostModerate();
        assertThat("StatusPostModerate = No", bannerStatusPostModerate, equalTo(Status.NO));
    }
}
