package ru.yandex.autotests.direct.api.ads.update;

import com.yandex.direct.api.v5.ads.AgeLabelEnum;
import org.json.simple.JSONObject;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.banners.AgeLabel;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Arrays.stream;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.joining;
import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.apache.commons.lang3.StringUtils.uncapitalize;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error.java;

/**
 * Created by semkagtn on 13.04.15.
 * https://st.yandex-team.ru/TESTIRT-5113
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Stories(ApiStories.JSON5)
@Description("Валидация JSON-запроса на обновление объяления")
@SuppressWarnings("unchecked")
public class UpdateJsonValidationTest {

    private static final String LOGIN = AdsLogins.CLIENT;
    private static final String PARAMS = "params";

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).protocol(ProtocolType.JSON).as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Before
    public void addUnitsToClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    @Description("Несуществующее поле в структуре AdUpdateItem")
    public void unknownFieldInAdUpdateItem() {
        String unknownField = uncapitalize(AdUpdateItemMap.TEXT_AD);

        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();
        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), new TextAdUpdateMap().defaultAd().getBean());
        adUpdateItem.put(unknownField, "Hello world!");
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000, Api5ErrorDetailsJava.ELEMENT_OF_ARRAY_CONTAIN_UNKNOWN_FIELD,
                        capitalize(UpdateRequestMap.ADS), unknownField))

                        .perl(new Api5Error(8000, Api5ErrorDetails.ELEMENT_OF_ARRAY_CONTAIN_UNKNOWN_FIELD,
                                capitalize(UpdateRequestMap.ADS), unknownField))
        );
    }

    @Test
    @Description("Не массив в поле Ads")
    public void notAnArrayInAdsField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();
        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), new TextAdUpdateMap().defaultAd().getBean());
        request.put(capitalize(UpdateRequestMap.ADS), adUpdateItem);

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                new Api5Error(8000, Api5ErrorDetails.FIELD_MUST_CONTAIN_ARRAY, capitalize(UpdateRequestMap.ADS)));
    }

    @Test
    @Description("Запрос без поля Ads")
    public void requestWithoutAdsField() {
        JSONObject request = new JSONObject();
        request.put(capitalize(UpdateRequestMap.ADS), null);

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        path(UpdateRequestMap.ADS)))

                        .perl(new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                                PARAMS, path(UpdateRequestMap.ADS)))
        );
    }

    @Test
    @Description("Массив внутри массива Ads")
    public void arrayInAdsFieldArray() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();
        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), new TextAdUpdateMap().defaultAd().getBean());
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(singletonList(adUpdateItem)));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000, Api5ErrorDetailsJava.INVALID_VALUE, path(UpdateRequestMap.ADS)))

                        .perl(new Api5Error(8000, Api5ErrorDetails.ELEMENT_OF_ARRAY_MUST_CONTAIN_OBJECT,
                                capitalize(UpdateRequestMap.ADS)))
        );
    }

    @Test
    @Description("Массив в поле Ads.TextAd")
    public void anArrayInTextAdField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), singletonList("text"));
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "title");

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), singletonList(textAdUpdate));
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_NOT_CONTAIN_ARRAY,
                        path(UpdateRequestMap.ADS, AdUpdateItemMap.TEXT_AD)))

                        .perl(new Api5Error(8000, Api5ErrorDetails.FIELD_MUST_NOT_CONTAIN_ARRAY,
                                capitalize(AdUpdateItemMap.TEXT_AD)))
        );
    }

    @Test
    @Description("Массив в поле Ads.TextAd.Text")
    public void anArrayInTextAdTextField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), singletonList("text"));
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "title");

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_NOT_CONTAIN_ARRAY,
                        path(UpdateRequestMap.ADS, AdUpdateItemMap.TEXT_AD, TextAdUpdateMap.TEXT)))

                        .perl(new Api5Error(8000, Api5ErrorDetails.FIELD_MUST_NOT_CONTAIN_ARRAY,
                                capitalize(TextAdUpdateMap.TEXT)))
        );
    }

    @Test
    @Ignore("DIRECT-88262")
    @Description("null в поле Ads.TextAd.Text")
    public void nullInTextAdTextField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), "null");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "title");

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                new Api5Error(8000, Api5ErrorDetails.CANNOT_HAVE_NULL_VALUE,
                        capitalize(TextAdUpdateMap.TEXT)));
    }

    @Test
    @Description("Массив в поле Ads.TextAd.Title")
    public void anArrayInTextAdTitleField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), "text");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), singletonList("title"));

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_NOT_CONTAIN_ARRAY,
                        path(UpdateRequestMap.ADS, AdUpdateItemMap.TEXT_AD, TextAdUpdateMap.TITLE)))

                        .perl(new Api5Error(8000, Api5ErrorDetails.FIELD_MUST_NOT_CONTAIN_ARRAY,
                                capitalize(TextAdUpdateMap.TITLE)))
        );
    }

    @Test
    @Description("Массив в поле Ads.TextAd.Title2")
    public void anArrayInTextAdTitle2Field() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), "text");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "title");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE_2), singletonList("title"));

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000,
                        Api5ErrorDetailsJava.FIELD_MUST_NOT_CONTAIN_ARRAY,
                        path(UpdateRequestMap.ADS, AdUpdateItemMap.TEXT_AD, TextAdUpdateMap.TITLE_2)))

                        .perl(new Api5Error(8000,
                                Api5ErrorDetails.FIELD_MUST_NOT_CONTAIN_ARRAY, path(TextAdUpdateMap.TITLE_2)))
        );
    }

    @Test
    @Ignore("DIRECT-88262")
    @Description("null в поле Ads.TextAd.Title")
    public void nullInTextAdTitleField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), "text");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "null");

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                new Api5Error(8000, Api5ErrorDetails.CANNOT_HAVE_NULL_VALUE, path(TextAdUpdateMap.TITLE)));
    }

    @Test
    @Description("Массив в поле Ads.TextAd.Href")
    public void anArrayInTextAdHrefField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), "text");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "title");
        textAdUpdate.put(capitalize(TextAdUpdateMap.HREF), singletonList("http://ya.ru"));

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000,
                        Api5ErrorDetailsJava.FIELD_MUST_NOT_CONTAIN_ARRAY,
                        path(UpdateRequestMap.ADS, AdUpdateItemMap.TEXT_AD, TextAdUpdateMap.HREF)))

                        .perl(new Api5Error(8000,
                                Api5ErrorDetails.FIELD_MUST_NOT_CONTAIN_ARRAY, path(TextAdUpdateMap.HREF)))
        );
    }

    @Test
    @Description("Массив в поле Ads.TextAd.VCardId")
    public void anArrayInTextAdVCardIdField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), "text");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "title");
        textAdUpdate.put(capitalize(TextAdUpdateMap.VCARD_ID), singletonList(1));

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_NOT_CONTAIN_ARRAY,
                        path(UpdateRequestMap.ADS, AdUpdateItemMap.TEXT_AD, TextAdUpdateMap.VCARD_ID)))

                        .perl(new Api5Error(8000, Api5ErrorDetails.FIELD_MUST_NOT_CONTAIN_ARRAY,
                                path(TextAdUpdateMap.VCARD_ID)))
        );
    }

    @Test
    @Description("Строка в поле Ads.TextAd.VCardId")
    public void stringInTextAdVCardIdField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), "text");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "title");
        textAdUpdate.put(capitalize(TextAdUpdateMap.VCARD_ID), "invalidValue");

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_INTEGER_VALUE,
                        path(UpdateRequestMap.ADS, AdUpdateItemMap.TEXT_AD, TextAdUpdateMap.VCARD_ID)))

                        .perl(new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_INTEGER_VALUE,
                                path(TextAdUpdateMap.VCARD_ID)))
        );
    }

    @Test
    @Description("Массив в поле Ads.TextAd.AdImageHash")
    public void anArrayInTextAdAdImageHashField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), "text");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "title");
        textAdUpdate.put(capitalize(TextAdUpdateMap.ADIMAGE_HASH), singletonList("someHash"));

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_NOT_CONTAIN_ARRAY,
                        path(UpdateRequestMap.ADS, AdUpdateItemMap.TEXT_AD, TextAdUpdateMap.ADIMAGE_HASH)))

                        .perl(new Api5Error(8000, Api5ErrorDetails.FIELD_MUST_NOT_CONTAIN_ARRAY,
                                capitalize(TextAdUpdateMap.ADIMAGE_HASH)))
        );
    }

    @Test
    @Description("Массив в поле Ads.TextAd.SitelinksSetId")
    public void anArrayInTextAdSitelinksSetIdField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), "text");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "title");
        textAdUpdate.put(capitalize(TextAdUpdateMap.SITELINK_SET_ID), singletonList(1));

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000, Api5ErrorDetails.FIELD_MUST_NOT_CONTAIN_ARRAY,
                        path(UpdateRequestMap.ADS, AdUpdateItemMap.TEXT_AD, TextAdUpdateMap.SITELINK_SET_ID)))

                        .perl(new Api5Error(8000, Api5ErrorDetails.FIELD_MUST_NOT_CONTAIN_ARRAY,
                                capitalize(TextAdUpdateMap.SITELINK_SET_ID)))
        );
    }

    @Test
    @Description("Строка в поле Ads.TextAd.SitelinksSetId")
    public void stringInTextAdSitelinksSetIdField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), "text");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "title");
        textAdUpdate.put(capitalize(TextAdUpdateMap.SITELINK_SET_ID), "invalidValue");

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_INTEGER_VALUE,
                        path(UpdateRequestMap.ADS, AdUpdateItemMap.TEXT_AD, TextAdUpdateMap.SITELINK_SET_ID)))

                        .perl(new Api5Error(8000, Api5ErrorDetails.FIELD_MUST_CONTAIN_INTEGER_VALUE,
                                capitalize(TextAdUpdateMap.SITELINK_SET_ID)))
        );
    }

    @Test
    @Description("Массив в поле Ads.TextAd.AgeLabel")
    public void anArrayInTextAdAgeLabelField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), "text");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "title");
        textAdUpdate.put(capitalize(TextAdUpdateMap.AGE_LABEL), singletonList(AgeLabel.MONTHS0));

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_NOT_CONTAIN_ARRAY,
                        path(UpdateRequestMap.ADS, AdUpdateItemMap.TEXT_AD, TextAdUpdateMap.AGE_LABEL)))

                        .perl(new Api5Error(8000,
                                Api5ErrorDetails.FIELD_MUST_NOT_CONTAIN_ARRAY, capitalize(TextAdUpdateMap.AGE_LABEL)))
        );
    }

    @Test
    @Description("Недопустимое значение в поле Ads.TextAd.AgeLabel")
    public void invalidValueInTextAdAgeLabelField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), "text");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "title");
        textAdUpdate.put(capitalize(TextAdUpdateMap.AGE_LABEL), "invalidValue");

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000,
                        Api5ErrorDetailsJava.INCORRECT_ENUMERATE_VALUE,
                        path(UpdateRequestMap.ADS, AdUpdateItemMap.TEXT_AD, TextAdUpdateMap.AGE_LABEL),
                        stream(AgeLabelEnum.values()).map(AgeLabelEnum::value).collect(joining(", "))))

                        .perl(new Api5Error(8000,
                                Api5ErrorDetails.INCORRECT_ENUMERATE_VALUE, capitalize(TextAdUpdateMap.AGE_LABEL)))
        );
    }

    @Test
    @Ignore("DIRECT-88262")
    @Description("null в поле Ads.TextAd.AgeLabel")
    public void nullInTextAdAgeLabelField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), "text");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "title");
        textAdUpdate.put(capitalize(TextAdUpdateMap.AGE_LABEL), "null");

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                new Api5Error(8000, Api5ErrorDetails.CANNOT_HAVE_NULL_VALUE, capitalize(TextAdUpdateMap.AGE_LABEL)));
    }

    @Test
    @Description("Несуществующее поле в структуре TextAdUpdate")
    public void unknownFieldInTextAdUpdate() {
        String unknownValue = uncapitalize(TextAdUpdateMap.AGE_LABEL);

        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();

        JSONObject textAdUpdate = new JSONObject();
        textAdUpdate.put(capitalize(TextAdUpdateMap.TEXT), "text");
        textAdUpdate.put(capitalize(TextAdUpdateMap.TITLE), "title");
        textAdUpdate.put(unknownValue, "someValue");

        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), 0L);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), textAdUpdate);
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000, Api5ErrorDetailsJava.ELEMENT_OF_ARRAY_CONTAIN_UNKNOWN_FIELD,
                        capitalize(UpdateRequestMap.ADS),
                        capitalize(AdUpdateItemMap.TEXT_AD) + "." + unknownValue))
                        .perl(new Api5Error(8000, Api5ErrorDetails.FIELD_CONTAIN_UNKNOWN_FIELD,
                                capitalize(AdUpdateItemMap.TEXT_AD), unknownValue))
        );
    }

    @Test
    @Description("Запрос на обновление текстового объявления без поля Ads.TextAd.Id")
    public void requestWithoutIdField() {
        JSONObject request = new JSONObject();
        JSONObject adUpdateItem = new JSONObject();
        adUpdateItem.put(capitalize(AdUpdateItemMap.ID), null);
        adUpdateItem.put(capitalize(AdUpdateItemMap.TEXT_AD), new TextAdUpdateMap().defaultAd().getBean());
        request.put(capitalize(UpdateRequestMap.ADS), singletonList(adUpdateItem));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(request,
                java(new Api5Error(8000, Api5ErrorDetailsJava.IN_ELEMENT_OF_ARRAY_MISSING_REQUIRED_FIELD,
                        capitalize(UpdateRequestMap.ADS), capitalize(AdUpdateItemMap.ID)))

                        .perl(new Api5Error(8000, Api5ErrorDetails.IN_ELEMENT_OF_ARRAY_MISSING_REQUIRED_FIELD,
                                capitalize(UpdateRequestMap.ADS), capitalize(AdUpdateItemMap.ID)))
        );
    }
}
