package ru.yandex.autotests.direct.api.ads.update;

import java.util.List;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error.java;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 14.04.15.
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Stories(ApiStories.UNITS)
@Description("Проверка списания баллов при обновлении объявлений")
public class UpdateUnitsTest {

    private static final String LOGIN = AdsLogins.UPDATE_UNITS;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Before
    public void addUnitsToUser() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    private static final int CALL_COST = 20;
    private static final int SET_COST = 20;
    private static final int ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR_COST = 50;

    private static long firstAdID;
    private static long secondAdID;

    @BeforeClass
    public static void createAds() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        long adGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        List<Long> adIds = api.userSteps.adsSteps().addDefaultTextAds(adGroupID, 2);
        firstAdID = adIds.get(0);
        secondAdID = adIds.get(1);
    }

    @Before
    public void addUnitsToClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    @Description("Снятие баллов при обновлении одного валидного объявления")
    public void updateOneAd() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(firstAdID)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd())));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + SET_COST;
        assertThat("снялось верное количество баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Снятие баллов при обновлении двух валидных объявлений")
    public void updateTwoAds() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap().withAds(
                new AdUpdateItemMap()
                        .withId(firstAdID)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd()),
                new AdUpdateItemMap()
                        .withId(secondAdID)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd())));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + 2 * SET_COST;
        assertThat("снялось верное количество баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Снятие баллов при обновлении одного невалидного объявления")
    public void updateOneInvalidAd() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(0L)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd())));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST;
        assertThat("снялось верное количество баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Снятие баллов при обновлении двух невалидных объявлений")
    public void updateTwoInvalidAds() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap().withAds(
                new AdUpdateItemMap()
                        .withId(0L)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd()),
                new AdUpdateItemMap()
                        .withId(-1L)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd())));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + 2 * ERROR_COST;
        assertThat("снялось верное количество баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Снятие баллов при обновлении одного валидного и одного невалидного объявлений")
    public void updateOneValidAndOneInvalidAds() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap().withAds(
                new AdUpdateItemMap()
                        .withId(firstAdID)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd()),
                new AdUpdateItemMap()
                        .withId(0L)
                        .withTextAd(new TextAdUpdateMap()
                                .defaultAd())));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + SET_COST + ERROR_COST;
        assertThat("снялось верное количество баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Невалидный запрос при обновлении объявлений")
    public void invalidRequest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.adsSteps().expectErrorOnAdsUpdate(new UpdateRequestMap().withAds((AdUpdateItemMap[]) null),
                java(new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_NOT_LESS_ELEMENTS,
                        capitalize(UpdateRequestMap.ADS), 1L))

                        .perl(new Api5Error(8000, Api5ErrorDetails.FIELD_MUST_CONTAIN_NOT_LESS_ELEMENTS,
                                capitalize(UpdateRequestMap.ADS)))
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        assertThat("снялось верное количество баллов", actualCost, equalTo(COMMON_REQUEST_ERROR_COST));
    }

    @Test
    @Description("Запрос на обновление объявления, когда у клиента ноль баллов на счету")
    public void requestWhenClientHasZeroUnits() {
        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(LOGIN, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assumeThat("у клиента нету баллов", units, equalTo(0));

        api.userSteps.adsSteps().expectErrorOnAdsUpdate(new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(firstAdID)
                                .withTextAd(new TextAdUpdateMap()
                                        .defaultAd())),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
    }
}
