package ru.yandex.autotests.direct.api.ads.update.cpmbanneradbuilderad;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.util.streamex.StreamEx;

import com.google.common.collect.ImmutableList;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListRuleOperatorEnum;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListTypeEnum;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang3.RandomUtils;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.ClientPixelProvidersCriterionType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.darkside.model.CampaignsType;
import ru.yandex.autotests.directapi.model.PixelProvider;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.CpmBannerAdBuilderAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.CpmBannerAdBuilderAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListAddItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListRuleArgumentItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListRuleItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Collections.singletonList;
import static ru.yandex.autotests.directapi.model.AdfoxDealPlacement.getPlacementJsonsFromPageIds;
import static ru.yandex.autotests.directapi.model.api5.general.ExpectedResult.success;

/**
 * https://st.yandex-team.ru/DIRECT-84893
 */
@Aqua.Test
@Features(AdsFeatures.ADD)
@Description("Проверка валидации пикселей аудита")
@Issue("https://st.yandex-team.ru/DIRECT-84893")
@RunWith(Parameterized.class)
public class UpdateCpmAdPixelValidationTest {

    private static final String clientLogin = AdsLogins.SUBCLIENT;
    private static final String agencyLogin = AdsLogins.AGENCY;
    private static final Long PUBLIC_GOAL_ID = 2499000002L;
    private static final Long PRIVATE_GOAL_ID = 2499000010L;
    public static final Long MIN_TEST_DEAL_ID = 2000000L;
    public static final Long MAX_TEST_DEAL_ID = 10000000L;
    public static final Long BIG_PLACEMENT_PAGE_ID = 63545390L;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(agencyLogin);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long rsyaCampaignId;
    private static Long foreignInventoryCampaignId;
    private static Long unknownInventoryCampaignId;

    @Parameterized.Parameter()
    public String description;

    @Parameterized.Parameter(1)
    public List<TestData> testDataList;

    private static Long creativeId;


    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {"Add unknown inventory positive test", singletonList(getUnknownInventoryPositiveTest())},
                {"Add public ya audience with only private permissions negative test",
                        ImmutableList.of(getTestDataYaAudiencePermissionsNegativeTest(),
                                getTestDataYaAudiencePermissionsPositiveTest())},
                {"YaAudience limits positive test",
                        singletonList(getTestDataForPixelYaAudienceLimitsPositiveTest())},
                {"YaAudience limits negative test",
                        singletonList(getTestDataForPixelYaAudienceLimitsNegativeTest())}
        });
    }

    @BeforeClass
    @Step("Подготовка данных")
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(agencyLogin);
        rsyaCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(clientLogin);
        api.userSteps.campaignFakeSteps().setType(rsyaCampaignId, CampaignsType.CPM_BANNER);
        foreignInventoryCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(clientLogin);
        api.userSteps.campaignFakeSteps().setType(foreignInventoryCampaignId, CampaignsType.CPM_DEALS);
        unknownInventoryCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(clientLogin);
        api.userSteps.campaignFakeSteps().setType(unknownInventoryCampaignId, CampaignsType.CPM_DEALS);
        addForeignPlacements(foreignInventoryCampaignId);
        addUnknownInventory(unknownInventoryCampaignId);

        creativeId = api.userSteps.getDirectJooqDbSteps().useShardForLogin(clientLogin).perfCreativesSteps()
                .saveDefaultCanvasCreativesForClient(Long.parseLong(User.get(clientLogin).getClientID()));
    }

    private static void addForeignPlacements(Long cpmDealCampaignId) {
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(agencyLogin)
                .placementsSteps()
                .addPlacement(BIG_PLACEMENT_PAGE_ID, 0);
        Long randomDealId = RandomUtils.nextLong(MIN_TEST_DEAL_ID, MAX_TEST_DEAL_ID);
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(agencyLogin)
                .adfoxSteps()
                .addDeal(randomDealId, Long.parseLong(User.get(agencyLogin).getClientID()),
                        getPlacementJsonsFromPageIds(singletonList(BIG_PLACEMENT_PAGE_ID)));
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(agencyLogin)
                .dealsSteps()
                .addDeal(randomDealId, Long.parseLong(User.get(agencyLogin).getClientID()));
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(clientLogin)
                .campaignDealsSteps()
                .addCampaignDeal(randomDealId, cpmDealCampaignId, 0);
    }

    private static void addUnknownInventory(Long cpmDealCampaignId) {
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(agencyLogin)
                .placementsSteps()
                .addPlacement(BIG_PLACEMENT_PAGE_ID + 1, 0);
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(agencyLogin)
                .placementsSteps()
                .addPlacement(BIG_PLACEMENT_PAGE_ID + 2, 1);
        Long randomDealId = RandomUtils.nextLong(MIN_TEST_DEAL_ID, MAX_TEST_DEAL_ID);
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(agencyLogin)
                .adfoxSteps()
                .addDeal(randomDealId, Long.parseLong(User.get(agencyLogin).getClientID()),
                        getPlacementJsonsFromPageIds(
                                ImmutableList.of(BIG_PLACEMENT_PAGE_ID + 1, BIG_PLACEMENT_PAGE_ID + 2)));
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(clientLogin)
                .dealsSteps()
                .addDeal(randomDealId, Long.parseLong(User.get(clientLogin).getClientID()));
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(clientLogin)
                .campaignDealsSteps()
                .addCampaignDeal(randomDealId, cpmDealCampaignId, 0);
    }

    @Test
    @Description("Попытка обновить баннер с пикселями")
    public void testBannerWithPixelsUpdate() {
        for (TestData testData : testDataList) {
            writeToClientPixelProviders(testData);
        }

        List<AdUpdateItemMap> adUpdateItemMaps = StreamEx.of(testDataList)
                .map(testData -> cpmAdUpdateItemMapByAdGroupAndPixel(addDefaultCpmAd(testData),
                        StreamEx.of(testData.pixelProviders).map(t -> t.testUrl()).toList()))
                .collect(Collectors.toList());


        if (testDataList.size() == 1) {
            api.userSteps.adsSteps()
                    .shouldGetResultOnUpdate(new UpdateRequestMap().withAds(adUpdateItemMaps), clientLogin,
                            testDataList.get(0).expectedResult);
        } else {
            api.userSteps.adsSteps()
                    .shouldGetResultOnUpdate(new UpdateRequestMap().withAds(adUpdateItemMaps),
                            clientLogin,
                            testDataList.get(0).expectedResult,
                            testDataList.get(1).expectedResult);
        }

        deleteFromClientPixelProviders();
    }

    private Long addDefaultCpmAd(TestData testData) {
        Long adGroupId = testData.adGroupIds.get();
        if (testData.campaignType == CampaignsType.CPM_DEALS) {
            return addDefaultCpmDealAd(adGroupId);
        } else {
            return addDefaultCpmBannerAd(adGroupId);
        }
    }

    private Long addDefaultCpmDealAd(Long adGroupId) {
        AdAddItemMap adAddItemMap = new AdAddItemMap()
                .withCpmBannerAdBuilderAd(new CpmBannerAdBuilderAdAddMap()
                        .withHref(CpmBannerAdBuilderAdAddMap.DEFAULT_HREF)
                        .withCreative(creativeId)
                        .withTrackingPixels(singletonList(PixelProvider.ADFOX.testUrl())))
                .withAdGroupId(adGroupId);
        return api.userSteps.adsSteps().addAd(adAddItemMap, clientLogin);
    }

    private Long addDefaultCpmBannerAd(Long adGroupId) {
        AdAddItemMap adAddItemMap = new AdAddItemMap()
                .withCpmBannerAdBuilderAd(new CpmBannerAdBuilderAdAddMap()
                        .withHref(CpmBannerAdBuilderAdAddMap.DEFAULT_HREF)
                        .withCreative(creativeId)
                        .withTrackingPixels(singletonList(PixelProvider.YANDEXAUDIENCE.testUrl())))
                .withAdGroupId(adGroupId);
        return api.userSteps.adsSteps().addAd(adAddItemMap, clientLogin);
    }

    private AdUpdateItemMap cpmAdUpdateItemMapByAdGroupAndPixel(Long adId, List<String> trackingPixels) {
        return new AdUpdateItemMap()
                .withCpmBannerAdBuilderAd(new CpmBannerAdBuilderAdUpdateMap()
                        .withTrackingPixels(trackingPixels))
                .withId(adId);
    }

    private void writeToClientPixelProviders(TestData testData) {
        ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsType campaignsTypeForDb =
                testData.campaignType == CampaignsType.CPM_DEALS ?
                        ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsType.cpm_deals :
                        ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsType.cpm_banner;
        for (int i = 0; i < testData.pixelProviders.size(); ++i) {
            if (testData.shouldAddPermissionsToDb.get(i)) {
                api.userSteps.getDirectJooqDbSteps()
                        .useShardForLogin(clientLogin)
                        .clientPixelProvidersSteps()
                        .addClientPixelProvider(Long.parseLong(User.get(clientLogin).getClientID()),
                                campaignsTypeForDb,
                                testData.inventoryTypeToDb,
                                testData.pixelProviders.get(i).pixelProviderFromDb());
            }
        }
    }

    private void deleteFromClientPixelProviders() {
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(clientLogin)
                .clientPixelProvidersSteps()
                .deleteClientProviders(Long.parseLong(User.get(clientLogin).getClientID()));
    }

    private static Long adGroupForRsyaTest() {
        return api.userSteps.adGroupsSteps().addDefaultCpmBannerUserProfileAdGroup(rsyaCampaignId, clientLogin);
    }

    private static Long getForeignInventoryPrivateSegmentsAdGroup() {
        return getForeignInventoryAdGroupWithSingleGoal(PRIVATE_GOAL_ID);
    }

    private static Long getForeignInventoryPublicSegmentsAdGroup() {
        return getForeignInventoryAdGroupWithSingleGoal(PUBLIC_GOAL_ID);
    }

    private static Long getForeignInventoryAdGroupWithSingleGoal(Long goalId) {
        Long adGroupId =
                api.userSteps.adGroupsSteps()
                        .addDefaultCpmBannerUserProfileAdGroup(foreignInventoryCampaignId, clientLogin);
        RetargetingListAddItemMap retargetingListAddItemMap = new RetargetingListAddItemMap()
                .withType(RetargetingListTypeEnum.AUDIENCE)
                .withName("Name" + RandomStringUtils.randomAlphabetic(3))
                .withDescription("Description" + RandomStringUtils.randomAlphabetic(3))
                .withRules(new RetargetingListRuleItemMap()
                        .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                .withMembershipLifeSpan(RandomUtils.nextInt(1, 90))
                                .withExternalId(goalId))
                        .withOperator(RetargetingListRuleOperatorEnum.ANY));
        Long retargetingListId = api.userSteps.retargetingListsSteps().add(clientLogin,
                new ru.yandex.autotests.directapi.model.api5.retargetinglists.AddRequestMap()
                        .withRetargetingLists(retargetingListAddItemMap)).get(0);
        api.userSteps.audienceTargetsSteps().addWithRetargetingList(clientLogin, adGroupId, retargetingListId);
        return adGroupId;
    }

    private static Long getUnknownInventoryAdGroup() {
        return api.userSteps.adGroupsSteps()
                .addDefaultCpmBannerUserProfileAdGroup(unknownInventoryCampaignId, clientLogin);
    }

    private static TestData getUnknownInventoryPositiveTest() {
        return new TestData(ImmutableList.of(PixelProvider.ADRIVER),
                (Supplier<Long>) () -> getUnknownInventoryAdGroup(),
                success(),
                ImmutableList.of(true),
                ClientPixelProvidersCriterionType.yandex,
                CampaignsType.CPM_DEALS);
    }

    private static TestData getTestDataYaAudiencePermissionsNegativeTest() {
        return new TestData(ImmutableList.of(PixelProvider.YANDEXAUDIENCE),
                (Supplier<Long>) () -> getForeignInventoryPrivateSegmentsAdGroup(),
                ExpectedResult.errors(new Notification(5004,
                        Api5ErrorDetailsJava.NO_RIGHTS_TO_AUDIENCE_PIXEL,
                        PixelProvider.YANDEXAUDIENCE.testUrl())),
                ImmutableList.of(false),
                ClientPixelProvidersCriterionType.private_,
                CampaignsType.CPM_DEALS);
    }

    private static TestData getTestDataYaAudiencePermissionsPositiveTest() {
        return new TestData(ImmutableList.of(PixelProvider.YANDEXAUDIENCE),
                (Supplier<Long>) () -> getForeignInventoryPublicSegmentsAdGroup(),
                success(),
                ImmutableList.of(true),
                ClientPixelProvidersCriterionType.public_,
                CampaignsType.CPM_DEALS);
    }

    private static TestData getTestDataForPixelYaAudienceLimitsPositiveTest() {
        return new TestData(
                ImmutableList.of(PixelProvider.YANDEXAUDIENCE),
                (Supplier<Long>) () -> adGroupForRsyaTest(),
                success(),
                ImmutableList.of(false),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.CPM_BANNER);
    }

    private static TestData getTestDataForPixelYaAudienceLimitsNegativeTest() {
        return new TestData(
                ImmutableList.of(PixelProvider.YANDEXAUDIENCE, PixelProvider.YANDEXAUDIENCE),
                (Supplier<Long>) () -> adGroupForRsyaTest(),
                ExpectedResult.errors(new Notification(6000,
                        Api5ErrorDetailsJava.MAX_YA_AUD_PIXELS_ON_BANNER)),
                ImmutableList.of(false, false),
                ClientPixelProvidersCriterionType.none,
                CampaignsType.CPM_BANNER);
    }

    private static class TestData {
        public List<PixelProvider> pixelProviders;
        public Supplier<Long> adGroupIds;
        public ExpectedResult expectedResult;
        public List<Boolean> shouldAddPermissionsToDb;
        public ClientPixelProvidersCriterionType inventoryTypeToDb;
        public CampaignsType campaignType;


        public TestData(
                List<PixelProvider> pixelProviders, Supplier<Long> adGroupIds,
                ExpectedResult expectedResult, List<Boolean> shouldAddPermissionsToDb,
                ClientPixelProvidersCriterionType inventoryTypeToDb,
                CampaignsType campaignType)
        {
            this.pixelProviders = pixelProviders;
            this.adGroupIds = adGroupIds;
            this.expectedResult = expectedResult;
            this.shouldAddPermissionsToDb = shouldAddPermissionsToDb;
            this.inventoryTypeToDb = inventoryTypeToDb;
            this.campaignType = campaignType;
        }
    }

}
