package ru.yandex.autotests.direct.api.ads.update.cpmbanneradbuilderad;

import java.time.LocalDateTime;
import java.util.Collections;

import org.glassfish.gmbal.Description;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersPerformanceStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusbssynced;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersPerformanceRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.darkside.model.CampaignsType;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.PixelProvider;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.CpmBannerAdBuilderAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static ru.yandex.autotests.directapi.darkside.model.Status.NO;

@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Description("Проверка обновления статусов (StatusBsSynced и StatusModerate) и LastChange")
@Issue("https://st.yandex-team.ru/DIRECT-83722")
public class UpdateCpmBannerAdBuilderAdStatusesTest {

    private static final String CLIENT = AdsLogins.CLIENT43;
    private static LocalDateTime previousLastChange = LocalDateTime.now().withNano(0).minusDays(1);
    private static DirectJooqDbSteps dbSteps;
    private static Long campaignId;
    private static Long adGroupId;
    private Long creativeId;
    private static Long adId;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    public static void prepareData() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        dbSteps = api.userSteps.getDarkSideSteps().getDirectJooqDbSteps().useShardForLogin(CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignFakeSteps().setType(campaignId, CampaignsType.CPM_BANNER);
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT)
                .campaignsSteps().setCampaignsStatusModerate(campaignId, CampaignsStatusmoderate.Yes);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultCpmBannerKeywordsAdGroup(campaignId);
        api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void setAdStatuses() {
        creativeId = dbSteps.perfCreativesSteps().saveDefaultCanvasCreativesForClient(
                Long.parseLong(User.get(CLIENT).getClientID()));
        adId = api.userSteps.adsSteps().addDefaultCpmBannerAdBuilderAd(adGroupId, creativeId);
        dbSteps.bannersPerformanceSteps()
                .setCreativeStatusModerate(campaignId, adGroupId, adId, BannersPerformanceStatusmoderate.Yes);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(adId);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adId, Status.YES);
        api.userSteps.bannersFakeSteps().setLastChange(adId, previousLastChange.toString());
    }

    @Test
    public void updateUrl() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withCpmBannerAdBuilderAd(
                                new CpmBannerAdBuilderAdUpdateMap().withHref("http://yandex.ru")
                        ).withId(adId)
                ),
                ExpectedResult.success()
        );

        checkBannerWithCreativeSentToModeration(adId, creativeId);
    }

    @Test
    public void updateCreative() {
        Long newCreativeId = dbSteps.perfCreativesSteps().saveDefaultCanvasCreativesForClient(
                Long.parseLong(User.get(CLIENT).getClientID()));
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withCpmBannerAdBuilderAd(
                                new CpmBannerAdBuilderAdUpdateMap().withCreative(newCreativeId)
                        ).withId(adId)
                ),
                ExpectedResult.success()
        );

        checkBannerWithCreativeSentToModeration(adId, newCreativeId);
    }


    @Test
    public void updateWithError() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withCpmBannerAdBuilderAd(
                                new CpmBannerAdBuilderAdUpdateMap().withHref("12345")
                        ).withId(adId)
                ),
                ExpectedResult.errors(
                        new Notification(5005, Api5ErrorDetails.INCORRECT_LINK)
                )
        );

        BannersRecord bannersRecord = dbSteps.bannersSteps().getBanner(adId);
        BannersPerformanceRecord bannersPerformance =
                dbSteps.bannersPerformanceSteps().getBannersPerformance(adId, creativeId);
        assertThat("в БД (banners_performance) верное значение StatusModerate",
                bannersPerformance.getStatusmoderate(), equalTo(BannersPerformanceStatusmoderate.Yes));
        assertThat("в БД (banners) верное значение StatusModerate",
                bannersRecord.getStatusmoderate(), equalTo(BannersStatusmoderate.Yes));
        assertThat("в БД (banners) верное значение StatusBsSynced",
                bannersRecord.getStatusbssynced(), equalTo(BannersStatusbssynced.Yes));
        assertThat("в БД (banners) верное значение LastChange",
                bannersRecord.getLastchange().toLocalDateTime(), equalTo(previousLastChange));
    }

    @Test
    public void updateRejected() {
        api.userSteps.bannersFakeSteps().setStatusModerate(adId, NO);
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withCpmBannerAdBuilderAd(
                                new CpmBannerAdBuilderAdUpdateMap().withHref("http://yandex.ru")
                        ).withId(adId)
                ),
                ExpectedResult.success()
        );

        checkBannerWithCreativeSentToModeration(adId, creativeId);
    }

    @Test
    @Issue("https://st.yandex-team.ru/DIRECT-96721")
    public void updateWithRejectedInGroup() {
        Long anotherCreativeId = dbSteps.perfCreativesSteps().saveDefaultCanvasCreativesForClient(
                Long.parseLong(User.get(CLIENT).getClientID()));
        Long anotherAdId = api.userSteps.adsSteps().addDefaultCpmBannerAdBuilderAd(adGroupId, anotherCreativeId);
        dbSteps.bannersPerformanceSteps()
                .setCreativeStatusModerate(campaignId, adGroupId, anotherAdId, BannersPerformanceStatusmoderate.Yes);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(anotherAdId);
        api.userSteps.bannersFakeSteps().setStatusModerate(anotherAdId, NO);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(anotherAdId, Status.YES);
        api.userSteps.bannersFakeSteps().setLastChange(anotherAdId, previousLastChange.toString());

        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withCpmBannerAdBuilderAd(
                                new CpmBannerAdBuilderAdUpdateMap().withHref("http://yandex.ru")
                        ).withId(adId)
                ),
                ExpectedResult.success()
        );

        checkBannerWithCreativeSentToModeration(adId, creativeId);

        BannersRecord bannersRecord = dbSteps.bannersSteps().getBanner(anotherAdId);
        BannersPerformanceRecord bannersPerformance =
                dbSteps.bannersPerformanceSteps().getBannersPerformance(anotherAdId, anotherCreativeId);
        assertThat("в БД (banners_performance) верное значение StatusModerate",
                bannersPerformance.getStatusmoderate(), equalTo(BannersPerformanceStatusmoderate.Yes));
        assertThat("в БД (banners) верное значение StatusModerate",
                bannersRecord.getStatusmoderate(), equalTo(BannersStatusmoderate.No));
        assertThat("в БД (banners) верное значение StatusBsSynced",
                bannersRecord.getStatusbssynced(), equalTo(BannersStatusbssynced.Yes));
        assertThat("в БД (banners) верное значение LastChange",
                bannersRecord.getLastchange().toLocalDateTime(), equalTo(previousLastChange));
    }

    private void checkBannerWithCreativeSentToModeration(Long adId, Long creativeId) {
        BannersRecord bannersRecord = dbSteps.bannersSteps().getBanner(adId);
        BannersPerformanceRecord bannersPerformance =
                dbSteps.bannersPerformanceSteps().getBannersPerformance(adId, creativeId);
        assertThat("в БД (banners_performance) верное значение StatusModerate",
                bannersPerformance.getStatusmoderate(), equalTo(BannersPerformanceStatusmoderate.Ready));
        assertThat("в БД (banners) верное значение StatusModerate",
                bannersRecord.getStatusmoderate(), equalTo(BannersStatusmoderate.Ready));
        assertThat("в БД (banners) верное значение StatusBsSynced",
                bannersRecord.getStatusbssynced(), equalTo(BannersStatusbssynced.No));
        assertThat("в БД (banners) верное значение LastChange",
                bannersRecord.getLastchange().toLocalDateTime(), greaterThan(previousLastChange));
    }


    @Test
    public void updateAddPixelsToCpmBanner() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withCpmBannerAdBuilderAd(
                                new CpmBannerAdBuilderAdUpdateMap().withTrackingPixels(
                                        Collections.singletonList(PixelProvider.ADFOX.testUrl()))
                        ).withId(adId)
                ),
                ExpectedResult.success()
        );

        BannersRecord bannersRecord = dbSteps.bannersSteps().getBanner(adId);
        assertThat("в БД (banners) верное значение StatusBsSynced",
                bannersRecord.getStatusbssynced(), equalTo(BannersStatusbssynced.No));
        assertThat("в БД (banners) верное значение StatusModerate",
                bannersRecord.getStatusmoderate(), equalTo(BannersStatusmoderate.Yes));
        assertThat("в БД (banners) верное значение LastChange",
                bannersRecord.getLastchange().toLocalDateTime(), greaterThan(previousLastChange));
    }
}
