package ru.yandex.autotests.direct.api.ads.update.mobileappad;

import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.yandex.direct.api.v5.ads.AddResponse;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersMobileContentRecord;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.darkside.enums.AdActionObjectType;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.common.Value;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by alexteut on 07.09.16.
 * https://st.yandex-team.ru/TESTIRT-10149
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Description("Проверка обновления действия (Action) при обновлении объявления")
@Issue("https://st.yandex-team.ru/DIRECT-54926")
@RunWith(Parameterized.class)
public class UpdateMobileAppAdActionsTest {

    private static final String LOGIN = AdsLogins.CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter
    public AdActionObjectType appAdAction;

    @Parameterized.Parameters(name = "appAdAction = {0}")
    public static Iterable<Object[]> data() {
        return Stream.of(AdActionObjectType.values())
                .map(x -> new Object[]{x})
                .collect(Collectors.toList());
    }

    private static int shard;
    private static long adGroupId;
    private long adId;

    @BeforeClass
    public static void prepareData() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);

        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        Long campaignId = api.userSteps.campaignSteps().addDefaultMobileAppCampaign();
        adGroupId = api.userSteps.adGroupsSteps().addDefaultMobileGroup(campaignId);
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void prepareAd() {
        AddResponse response = api.userSteps.adsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAds(new AdAddItemMap()
                                .withAdGroupId(adGroupId)
                                .withMobileAppAd(new MobileAppAdAddMap()
                                        .defaultMobileAppAd()
                                        .withAction(appAdAction.getMobileAppAdAction()))
                        ),
                ExpectedResult.success());
        adId = response.getAddResults().get(0).getId();

        api.userSteps.bannersFakeSteps().setStatusBsSynced(adId, Value.YES);

        BannerFakeInfo bannerFakeInfoBefore = api.userSteps.bannersFakeSteps().getBannerParams(adId);
        assertThat("статус синхронизации с БК до редактирования кнопок", bannerFakeInfoBefore.getStatusBsSynced(), equalTo(Value.YES));
    }

    @Test
    @Description("Проверка статуса BsSynced после обновления, которое изменило кнопки")
    public void updateActionsField() {
        AdActionObjectType exceptedActionObjectType = appAdAction == AdActionObjectType.DOWNLOAD
                ? AdActionObjectType.UPDATE : AdActionObjectType.DOWNLOAD;

        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adId)
                        .withMobileAppAd(new MobileAppAdUpdateMap()
                                .defaultMobileAppAd()
                                .withAction(exceptedActionObjectType.getMobileAppAdAction()))
                ), ExpectedResult.success()
        );

        BannersMobileContentRecord actualMobileContent = api.userSteps
                .getDirectJooqDbSteps()
                .useShard(shard)
                .mobileContentSteps()
                .getBannersMobileContent(adId);

        assertThat("активность объявления обновлась верно",
                actualMobileContent.getPrimaryAction().toString(),
                equalTo(exceptedActionObjectType.getPrimaryAction().toString()));
    }

    @Test
    @Description("Проверка статуса BsSynced после обновления")
    public void updateActionsFieldStatusBsSynced() {
        AdActionObjectType exceptedActionObjectType = appAdAction == AdActionObjectType.DOWNLOAD
                ? AdActionObjectType.UPDATE : AdActionObjectType.DOWNLOAD;

        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(adId)
                                .withMobileAppAd(new MobileAppAdUpdateMap()
                                        .defaultMobileAppAd()
                                        .withAction(exceptedActionObjectType.getMobileAppAdAction()))
                        ), ExpectedResult.success()
        );

        BannerFakeInfo bannerFakeInfoAfter = api.userSteps.bannersFakeSteps().getBannerParams(adId);
        assertThat("статус синхронизации с БК после редактирования кнопок", bannerFakeInfoAfter.getStatusBsSynced(), equalTo(Value.NO));
    }

    @Test
    @Description("Проверка статуса BsSynced после обновления, которое ничего не изменило")
    public void updateActionsFieldStatusBsSyncedSameAction() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(adId)
                                .withMobileAppAd(new MobileAppAdUpdateMap()
                                        .defaultMobileAppAd()
                                        .withAction(appAdAction.getMobileAppAdAction()))
                        ), ExpectedResult.success()
        );

        BannerFakeInfo bannerFakeInfoAfter = api.userSteps.bannersFakeSteps().getBannerParams(adId);
        assertThat("статус синхронизации с БК после редактирования кнопок", bannerFakeInfoAfter.getStatusBsSynced(), equalTo(Value.YES));
    }
}
