package ru.yandex.autotests.direct.api.ads.update.mobileappad;

import com.yandex.direct.api.v5.ads.MobileAppFeatureEnum;
import com.yandex.direct.api.v5.general.MobileAppAdActionEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersMobileContentPrimaryAction;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PhrasesAdgroupType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersMobileContentRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.enums.AdGroupType;
import ru.yandex.autotests.directapi.enums.ReflectedAttrs;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppAdFeatureItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasProperty;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 27.08.15.
 * https://st.yandex-team.ru/TESTIRT-6853
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Description("Проверка обновления значений в БД у объявлений с рекламой мобильных приложений")
@Issue("https://st.yandex-team.ru/DIRECT-44779")
public class UpdateMobileAppAdDatabaseValuesTest {

    private static final String LOGIN = AdsLogins.CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);
    private static DirectJooqDbSteps jooqDbSteps;

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static int shard;

    @BeforeClass
    public static void prepareStaticData() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
    }

    private long adId;

    @Before
    @Step("Подготовка данных для теста")
    public void prepareData() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);

        long mobileContentAdGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard);
        jooqDbSteps.adGroupsSteps().setType(mobileContentAdGroupId, PhrasesAdgroupType.mobile_content);

        adId = api.userSteps.adsSteps().addDefaultMobileAppAd(mobileContentAdGroupId);
    }

    @Test
    @Description("Проверка правильности обновления заголовка, описания и ссылки объявления" +
            " с рекламой мобильного приложения")
    public void checkTitleAndTextAndTrackingUrlOfMobileAppAd() {
        String title = "Некоторое название";
        String text = "Некоторое описание мобильного приложения";
        String trackingUrl = "http://appmetrika.yandex.ru?pid=yandexdirect_int&clickid={logid}";

        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adId)
                        .withMobileAppAd(new MobileAppAdUpdateMap()
                                .defaultMobileAppAd()
                                .withTitle(title)
                                .withText(text)
                                .withTrackingUrl(trackingUrl))));

        BannersRecord actualBanner = api.userSteps
                .getDirectJooqDbSteps()
                .useShard(shard)
                .bannersSteps()
                .getBanner(adId);

        assertThat("заголовок, описание и ссылка объявления обновились верно",
                actualBanner,
                allOf(
                        hasProperty("title", equalTo(title)),
                        hasProperty("body", equalTo(text)),
                        hasProperty("href", equalTo(trackingUrl))));
    }

    @Test
    @Description("Проверка правильности обновления аттрибутов объявления: аттрибуты включены")
    public void checkEnabledFeaturesOfMobileAppAd() {
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap().withAds(new AdUpdateItemMap()
                .withId(adId)
                .withMobileAppAd(new MobileAppAdUpdateMap()
                        .defaultMobileAppAd()
                        .withFeatures(
                                new MobileAppAdFeatureItemMap()
                                        .withEnabled(YesNoEnum.NO)
                                        .withFeature(MobileAppFeatureEnum.ICON),
                                new MobileAppAdFeatureItemMap()
                                        .withEnabled(YesNoEnum.YES)
                                        .withFeature(MobileAppFeatureEnum.PRICE),
                                new MobileAppAdFeatureItemMap()
                                        .withEnabled(YesNoEnum.YES)
                                        .withFeature(MobileAppFeatureEnum.RATINGS)))));

        BannersMobileContentRecord actualMobileContent = api.userSteps
                .getDirectJooqDbSteps()
                .useShard(shard)
                .mobileContentSteps()
                .getBannersMobileContent(adId);

        Assert.assertThat("атрибуты объявления обновились верно",
                actualMobileContent.getReflectedAttrs(),
                equalTo(String.join(",", ReflectedAttrs.PRICE.toString(), ReflectedAttrs.RATING_VOTES.toString())));
    }

    @Test
    @Description("Проверка правильности обновления аттрибутов объявления: один аттрибут включен, другой - выключен")
    public void checkDisabledFeaturesOfMobileAppAd() {
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap().withAds(new AdUpdateItemMap()
                .withId(adId)
                .withMobileAppAd(new MobileAppAdUpdateMap()
                        .defaultMobileAppAd()
                        .withFeatures(
                                new MobileAppAdFeatureItemMap()
                                        .withEnabled(YesNoEnum.NO)
                                        .withFeature(MobileAppFeatureEnum.ICON),
                                new MobileAppAdFeatureItemMap()
                                        .withEnabled(YesNoEnum.YES)
                                        .withFeature(MobileAppFeatureEnum.PRICE),
                                new MobileAppAdFeatureItemMap()
                                        .withEnabled(YesNoEnum.NO)
                                        .withFeature(MobileAppFeatureEnum.RATINGS)))));

        BannersMobileContentRecord actualMobileContent = api.userSteps
                .getDirectJooqDbSteps()
                .useShard(shard)
                .mobileContentSteps()
                .getBannersMobileContent(adId);

        Assert.assertThat("атрибуты объявления обновились верно",
                actualMobileContent.getReflectedAttrs(),
                equalTo(ReflectedAttrs.PRICE.toString()));
    }

    @Test
    @Description("Проверка правильности обновления действия объявления")
    public void checkActionOfMobileAppAd() {
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adId)
                        .withMobileAppAd(new MobileAppAdUpdateMap()
                                .defaultMobileAppAd()
                                .withAction(MobileAppAdActionEnum.DOWNLOAD))));

        BannersMobileContentRecord actualMobileContent = api.userSteps
                .getDirectJooqDbSteps()
                .useShard(shard)
                .mobileContentSteps()
                .getBannersMobileContent(adId);

        Assert.assertThat("действие объявления сохранились верно",
                actualMobileContent.getPrimaryAction(),
                equalTo(BannersMobileContentPrimaryAction.download));
    }

    @Test
    @Description("Проверка \"затирания\" значения поля TrackingUrl")
    public void checkNilTrackingUrlOfMobileAppAd() {
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adId)
                        .withMobileAppAd(new MobileAppAdUpdateMap()
                                .defaultMobileAppAd()
                                .withTrackingUrl(null))));

        BannersRecord actualBanner = api.userSteps
                .getDirectJooqDbSteps()
                .useShard(shard)
                .bannersSteps()
                .getBanner(adId);

        assumeThat("из БД было получено объявление", actualBanner, notNullValue());

        assertThat("заголовок, описание и ссылка объявления обновились верно",
                actualBanner.getHref(), nullValue());
    }
}
