package ru.yandex.autotests.direct.api.ads.update.mobileappad;

import com.yandex.direct.api.v5.ads.MobAppAgeLabelEnum;
import com.yandex.direct.api.v5.ads.MobileAppFeatureEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PhrasesAdgroupType;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppAdFeatureItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.core.IsNot.not;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by semkagtn on 30.09.15.
 * https://st.yandex-team.ru/TESTIRT-5113
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Description("Проверка обновления статусов (StatusBsSynced и StatusModerate) при обновлении объявления с РМП")
@Issue("https://st.yandex-team.ru/DIRECT-46329")
public class UpdateMobileAppAdStatusesTest {

    private static final String LOGIN = AdsLogins.CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);
    private static DirectJooqDbSteps jooqDbSteps;

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long campaignId;
    private static long adId;
    private static DateTime lastChange = DateTime.now().minusDays(1);
    private static final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");

    @BeforeClass
    public static void prepareData() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN)
                .campaignsSteps().setCampaignsStatusModerate(campaignId, CampaignsStatusmoderate.Yes);
        long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);

        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN);
        jooqDbSteps.adGroupsSteps().setType(adGroupId, PhrasesAdgroupType.mobile_content);

        adId = api.userSteps.adsSteps().addAd(new AdAddItemMap()
                .withAdGroupId(adGroupId)
                .withMobileAppAd(new MobileAppAdAddMap()
                        .defaultMobileAppAd()
                        .withAgeLabel(MobAppAgeLabelEnum.AGE_6)));
    }

    @Before
    @Step("Подготовка данных для теста")
    public void setAdStatuses() {
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(adId);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adId, Status.YES);
        api.userSteps.bannersFakeSteps().setLastChange(adId, lastChange.toString());

        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    @Description("Проверка обновления статусов при обновлении текста объявления")
    public void updateTextField() {
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adId)
                        .withMobileAppAd(new MobileAppAdUpdateMap()
                                .withText("NEW TEXT"))));

        BannerFakeInfo actualBannerFakeInfo = api.userSteps.bannersFakeSteps().getBannerParams(adId);
        BannerFakeInfo expectedBannerFakeInfo = new BannerFakeInfo();
        expectedBannerFakeInfo.setStatusBsSynced(Status.NO);
        expectedBannerFakeInfo.setStatusModerate(Status.READY);

        assertThat("статусы объявления обновлись верно",
                actualBannerFakeInfo, beanEquivalent(expectedBannerFakeInfo));

        LocalDate actualDate = formatter.parseLocalDate(actualBannerFakeInfo.getLastChange());
        assertThat("LastChange объявления обновился", actualDate, equalTo(LocalDate.now()));
    }

    @Test
    @Description("Проверка обновления статусов при обновлении заголовка объявления")
    public void updateTitleField() {
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adId)
                        .withMobileAppAd(new MobileAppAdUpdateMap()
                                .withTitle("NEW TITLE"))));

        BannerFakeInfo actualBannerFakeInfo = api.userSteps.bannersFakeSteps().getBannerParams(adId);
        BannerFakeInfo expectedBannerFakeInfo = new BannerFakeInfo();
        expectedBannerFakeInfo.setStatusBsSynced(Status.NO);
        expectedBannerFakeInfo.setStatusModerate(Status.READY);

        assertThat("статусы объявления обновлись верно",
                actualBannerFakeInfo, beanEquivalent(expectedBannerFakeInfo));

        LocalDate actualDate = formatter.parseLocalDate(actualBannerFakeInfo.getLastChange());
        assertThat("LastChange объявления обновился", actualDate, equalTo(LocalDate.now()));
    }

    @Test
    @Description("Проверка обновления статусов при обновлении TrackingUrl объявления")
    public void updateTrackingUrlField() {
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adId)
                        .withMobileAppAd(new MobileAppAdUpdateMap()
                                .withTrackingUrl("http://ad.apps.fm?pid=yandexdirect_int&clickid={logid}"))));

        BannerFakeInfo actualBannerFakeInfo = api.userSteps.bannersFakeSteps().getBannerParams(adId);
        BannerFakeInfo expectedBannerFakeInfo = new BannerFakeInfo();
        expectedBannerFakeInfo.setStatusBsSynced(Status.NO);
        expectedBannerFakeInfo.setStatusModerate(Status.READY);

        assertThat("статусы объявления обновлись верно",
                actualBannerFakeInfo, beanEquivalent(expectedBannerFakeInfo));

        LocalDate actualDate = formatter.parseLocalDate(actualBannerFakeInfo.getLastChange());
        assertThat("LastChange объявления обновился", actualDate, equalTo(LocalDate.now()));
    }

    @Test
    @Description("Проверка обновления статусов при обновлении AgeLabel объявления")
    public void updateAgeLabelField() {
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adId)
                        .withMobileAppAd(new MobileAppAdUpdateMap()
                                .withAgeLabel(MobAppAgeLabelEnum.AGE_18))));

        BannerFakeInfo actualBannerFakeInfo = api.userSteps.bannersFakeSteps().getBannerParams(adId);
        BannerFakeInfo expectedBannerFakeInfo = new BannerFakeInfo();
        expectedBannerFakeInfo.setStatusBsSynced(Status.NO);
        expectedBannerFakeInfo.setStatusModerate(Status.YES);

        assertThat("статусы объявления обновлись верно",
                actualBannerFakeInfo, beanEquivalent(expectedBannerFakeInfo));

        LocalDate actualDate = formatter.parseLocalDate(actualBannerFakeInfo.getLastChange());
        assertThat("LastChange объявления обновился", actualDate, equalTo(LocalDate.now()));
    }

    @Test
    @Description("Проверка обновления статусов при обновлении Features объявления")
    public void updateFeaturesField() {
        api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap()
                .withAds(new AdUpdateItemMap()
                        .withId(adId)
                        .withMobileAppAd(new MobileAppAdUpdateMap()
                                .withFeatures(new MobileAppAdFeatureItemMap()
                                        .withEnabled(YesNoEnum.YES)
                                        .withFeature(MobileAppFeatureEnum.CUSTOMER_RATING)))));

        BannerFakeInfo actualBannerFakeInfo = api.userSteps.bannersFakeSteps().getBannerParams(adId);
        BannerFakeInfo expectedBannerFakeInfo = new BannerFakeInfo();
        expectedBannerFakeInfo.setStatusBsSynced(Status.NO);
        expectedBannerFakeInfo.setStatusModerate(Status.YES);

        assertThat("статусы объявления обновлись верно",
                actualBannerFakeInfo, beanEquivalent(expectedBannerFakeInfo));

        LocalDate actualDate = formatter.parseLocalDate(actualBannerFakeInfo.getLastChange());
        assertThat("LastChange объявления обновился", actualDate, equalTo(LocalDate.now()));
    }

    @Test
    @Description("Проверка необновления статусов при неудавшемся обновлении")
    public void updateWithError() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(adId)
                                .withMobileAppAd(new MobileAppAdUpdateMap()
                                        .withText(""))),
                ExpectedResult.errors(new Notification(5000,
                        Api5ErrorDetails.NO_VALUE_IN_FIELD, capitalize(MobileAppAdAddMap.TEXT))));

        BannerFakeInfo actualBannerFakeInfo = api.userSteps.bannersFakeSteps().getBannerParams(adId);
        BannerFakeInfo expectedBannerFakeInfo = new BannerFakeInfo();
        expectedBannerFakeInfo.setStatusBsSynced(Status.YES);
        expectedBannerFakeInfo.setStatusModerate(Status.YES);

        assertThat("статусы объявления обновились верно",
                actualBannerFakeInfo, beanEquivalent(expectedBannerFakeInfo));

        LocalDate actualDate = formatter.parseLocalDate(actualBannerFakeInfo.getLastChange());
        assertThat("LastChange объявления обновился", actualDate, not(equalTo(LocalDate.now())));
    }
}
