package ru.yandex.autotests.direct.api.ads.update.mobileappimagead;

import java.time.LocalDateTime;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusbssynced;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppImageAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.anyOf;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;

/**
 * Created by gerdler on 13.07.16.
 * https://st.yandex-team.ru/TESTIRT-9322
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Description("Проверка обновления статусов (StatusBsSynced и StatusModerate) и LastChange " +
        "при обновлении графического объявления РМП")
@Issue("https://st.yandex-team.ru/DIRECT-54754")
public class UpdateMobileAppImageAdStatusesTest {

    private static final String CLIENT = AdsLogins.CLIENT43;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long adId;
    private static LocalDateTime previousLastChange = LocalDateTime.now().withNano(0).minusDays(1);
    private static DirectJooqDbSteps dbSteps;

    @BeforeClass
    public static void prepareData() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);

        dbSteps = api.userSteps.getDarkSideSteps().getDirectJooqDbSteps().useShardForLogin(CLIENT);
        Long campaignId = api.userSteps.campaignSteps().addDefaultMobileAppCampaign();
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT)
                .campaignsSteps().setCampaignsStatusModerate(campaignId, CampaignsStatusmoderate.Yes);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultMobileGroup(campaignId);
        api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
        String adImageHash = api.userSteps.imagesStepsV5().addImageAdAdImage(5);
        adId = api.userSteps.adsSteps().addDefaultMobileAppImageAd(adGroupId, adImageHash);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void setAdStatuses() {
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(adId);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adId, Status.YES);
        api.userSteps.bannersFakeSteps().setLastChange(adId, previousLastChange.toString());
    }

    @Test
    @Description("Проверка обновления статусов при обновлении Href")
    public void updateHref() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withMobileAppImageAd(
                                new MobileAppImageAdUpdateMap().withTrackingUrl("http://appmetrica.yandex.com")
                        ).withId(adId)
                ),
                ExpectedResult.success()
        );

        BannersRecord bannersRecord = dbSteps.bannersSteps().getBanner(adId);
        assertThat("в БД верное значение StatusBsSynced",
                bannersRecord.getStatusbssynced(), equalTo(BannersStatusbssynced.No));
        assertThat("в БД верное значение StatusModerate",
                bannersRecord.getStatusmoderate(), anyOf(
                        equalTo(BannersStatusmoderate.Ready),
                        equalTo(BannersStatusmoderate.Sending)
                ));
        assertThat("в БД верное значение LastChange",
                bannersRecord.getLastchange().toLocalDateTime(), greaterThan(previousLastChange));
    }

    @Test
    @Description("Проверка обновления статусов при изменении изображения в объявлении")
    public void updateAdImageHash() {
        String newAdImageHash = api.userSteps.imagesStepsV5().addImageAdAdImage(6);
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withMobileAppImageAd(
                                new MobileAppImageAdUpdateMap().withAdImageHash(newAdImageHash)
                        ).withId(adId)
                ),
                ExpectedResult.success()
        );

        BannersRecord bannersRecord = dbSteps.bannersSteps().getBanner(adId);
        assertThat("в БД верное значение StatusBsSynced",
                bannersRecord.getStatusbssynced(), equalTo(BannersStatusbssynced.No));
        assertThat("в БД верное значение StatusModerate",
                bannersRecord.getStatusmoderate(), anyOf(
                        equalTo(BannersStatusmoderate.Ready),
                        equalTo(BannersStatusmoderate.Sending)
                ));
        assertThat("в БД верное значение LastChange",
                bannersRecord.getLastchange().toLocalDateTime(), greaterThan(previousLastChange));
    }


    @Test
    @Description("Проверка необновления статусов при неудавшемся обновлении")
    public void updateWithError() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withAds(
                        new AdUpdateItemMap().withMobileAppImageAd(
                                new MobileAppImageAdUpdateMap().withTrackingUrl("12345")
                        ).withId(adId)
                ),
                ExpectedResult.errors(
                        new Notification(5005, Api5ErrorDetails.INCORRECT_LINK)
                )
        );

        BannersRecord bannersRecord = dbSteps.bannersSteps().getBanner(adId);
        assertThat("в БД верное значение StatusBsSynced",
                bannersRecord.getStatusbssynced(), equalTo(BannersStatusbssynced.Yes));
        assertThat("в БД верное значение StatusModerate",
                bannersRecord.getStatusmoderate(), equalTo(BannersStatusmoderate.Yes));
        assertThat("в БД верное значение LastChange",
                bannersRecord.getLastchange().toLocalDateTime(), equalTo(previousLastChange));
    }
}
