package ru.yandex.autotests.direct.api.ads.update.textad;

import java.sql.Timestamp;
import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.GetResponse;
import com.yandex.direct.api.v5.ads.TextAdFieldEnum;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.api.ads.AdsStories;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannerImagesStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersPhoneflag;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusbssynced;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersStatussitelinksmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.GroupFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.enums.ImageType;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by semkagtn on 14.04.15.
 * https://st.yandex-team.ru/TESTIRT-5113
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Stories(AdsStories.TEXT_AD)
@Description("Проверка обновления картинки в текстовом объявлении")
@RunWith(Parameterized.class)
public class UpdateAdImageTest {

    private static final String LOGIN = AdsLogins.CLIENT11;
    private static final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");
    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(LOGIN);
    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();
    private static Long adGroupID;

    private static DateTime lastChange = DateTime.now().minusDays(3);
    @Rule
    public Trashman trasher = new Trashman(api);
    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public String oldAdImageHash;

    @Parameterized.Parameter(2)
    public String newAdImageHash;

    @Parameterized.Parameter(3)
    public ProtocolType protocol;

    @Parameterized.Parameter(4)
    public String expectedGroupStatus;

    @Parameterized.Parameter(5)
    public String expectedStatusBsSynced;

    @Parameterized.Parameter(6)
    public String expectedImageStatusModerate;

    @Parameterized.Parameter(7)
    public boolean expectedLastChangeUpdate;

    private long adID;

    @Parameterized.Parameters(name = "{0}, protocol = {3}")
    public static Collection<Object[]> data() {
        api.protocol(ProtocolType.JSON);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN).campaignsSteps()
                .setCampaignsStatusModerate(campaignID, CampaignsStatusmoderate.Yes);
        adGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);
        String[] adImageHashes = api.userSteps.imagesSteps().configLoginImages(LOGIN, ImageType.REGULAR, 6, 7);

        return Arrays.asList(new Object[][]{
                {"Привязка картинки к объявлению", null, adImageHashes[0], ProtocolType.JSON,
                        /* groupBsSync */ Status.NO, /* adBsSync */ Status.NO, /* imgModerate */ Status.READY,
                        /* lastChangeUpdate */ true},
                {"Изменение картинки в объявлении", adImageHashes[0], adImageHashes[1], ProtocolType.JSON,
                        /* groupBsSync */ Status.YES, /* adBsSync */ Status.NO, /* imgModerate */ Status.READY,
                        /* lastChangeUpdate */ true},
                {"Старая картинка в update", adImageHashes[0], adImageHashes[0], ProtocolType.JSON,
                        /* groupBsSync */ Status.YES, /* adBsSync */ Status.YES, /* imgModerate */ Status.YES,
                        /* lastChangeUpdate */ false},
                {"Отвязка картинки от объявления", adImageHashes[0], null, ProtocolType.JSON,
                        /* groupBsSync */ Status.YES, /* adBsSync */ Status.NO, /* imgModerate */ Status.YES,
                        /* lastChangeUpdate */ true},
                {"Отвязка картинки от объявления", adImageHashes[0], null, ProtocolType.SOAP,
                        /* groupBsSync */ Status.YES, /* adBsSync */ Status.NO, /* imgModerate */ Status.YES,
                        /* lastChangeUpdate */ true},
        });
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createAd() {
        api.protocol(ProtocolType.JSON);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        adID = api.userSteps.adsSteps().addAd(new AdAddItemMap()
                .withAdGroupId(adGroupID)
                .withTextAd(new TextAdAddMap()
                        .defaultTextAd()
                        .withAdImageHash(oldAdImageHash)));
    }

    @Test
    public void updateAdImage() {
        api.protocol(protocol);
        api.userSteps.adsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(adID)
                                .withTextAd(new TextAdUpdateMap()
                                        .withAdImageHash(newAdImageHash))),
                ExpectedResult.success(adID));

        api.protocol(ProtocolType.JSON);
        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withIds(adID))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.AD_IMAGE_HASH));
        assertThat("картинка в объявлении была успешно изменена",
                getResponse.getAds().get(0).getTextAd().getAdImageHash().getValue(), equalTo(newAdImageHash));
    }

    @Test
    @Issue("DIRECT-49882")
    public void checkStatusesAfterUpdate() {
        DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN);
        BannersRecord banner = jooqDbSteps.bannersSteps().getBanner(adID);
        banner.setStatusmoderate(BannersStatusmoderate.Yes);
        banner.setStatuspostmoderate(BannersStatuspostmoderate.Yes);
        banner.setStatussitelinksmoderate(BannersStatussitelinksmoderate.Yes);
        banner.setPhoneflag(BannersPhoneflag.Yes);
        banner.setStatusbssynced(BannersStatusbssynced.Yes);
        banner.setLastchange(new Timestamp(lastChange.getMillis()));
        jooqDbSteps.bannersSteps().updateBanners(banner);
        jooqDbSteps.imagesSteps().setBannerImagesStatusModerate(oldAdImageHash, BannerImagesStatusmoderate.Yes);

        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(adGroupID, Status.YES);
        api.protocol(protocol);
        api.userSteps.adsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(adID)
                                .withTextAd(new TextAdUpdateMap()
                                        .withAdImageHash(newAdImageHash))),
                ExpectedResult.success(adID));

        BannerFakeInfo actualBannerFakeInfo = api.userSteps.bannersFakeSteps().getBannerParams(adID);

        BannerFakeInfo expectedBannerFakeInfo = new BannerFakeInfo();
        expectedBannerFakeInfo.setStatusBsSynced(expectedStatusBsSynced);
        expectedBannerFakeInfo.setStatusModerate(Status.YES);
        expectedBannerFakeInfo.setImageStatusModerate(expectedImageStatusModerate);

        assertThat("статусы объявления обновлись верно",
                actualBannerFakeInfo, beanEquivalent(expectedBannerFakeInfo));

        LocalDate actualLastChange = formatter.parseLocalDate(actualBannerFakeInfo.getLastChange());
        LocalDate expectedLastChange = expectedLastChangeUpdate ? LocalDate.now() : lastChange.toLocalDate();
        assertThat("LastChange объявления обновился", actualLastChange, equalTo(expectedLastChange));

        GroupFakeInfo groupFakeInfo = api.userSteps.groupFakeSteps().getGroupParams(adGroupID);
        assertThat("у группы сбросился StatusBsSynced",
                groupFakeInfo.getStatusBsSynced(), equalTo(expectedGroupStatus));
    }
}
