package ru.yandex.autotests.direct.api.ads.update.textad;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.GetResponse;
import com.yandex.direct.api.v5.ads.PriceCurrencyEnum;
import com.yandex.direct.api.v5.ads.PriceQualifierEnum;
import com.yandex.direct.api.v5.ads.TextAdFieldEnum;
import com.yandex.direct.api.v5.ads.TextAdPriceExtensionFieldEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.matchers.beans.version5.BigDecimalDiffer;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.PriceExtensionAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.PriceExtensionUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.CompareStrategy;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static com.yandex.direct.api.v5.ads.PriceCurrencyEnum.EUR;
import static com.yandex.direct.api.v5.ads.PriceCurrencyEnum.RUB;
import static com.yandex.direct.api.v5.ads.PriceQualifierEnum.FROM;
import static com.yandex.direct.api.v5.ads.PriceQualifierEnum.NONE;
import static ru.yandex.autotests.direct.api.ads.update.textad.UpdateBannerPricePositiveTest.getMatcher;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.Api5CompareStrategies.allFields;
import static ru.yandex.autotests.directapi.model.api5.general.ExpectedResult.errors;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.beanfield.BeanFieldPath.newPath;

@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Description("Проверка работы поля BannerPrice: негативные кейсы")
@Issue("https://st.yandex-team.ru/DIRECT-89499")
@RunWith(Parameterized.class)
public class UpdateBannerPriceNegativeTest {

    private static final CompareStrategy COMPARE_STRATEGY = allFields().forFields(
            newPath("\\d+", "textAd", "priceExtension", "value", "price"),
            newPath("\\d+", "textAd", "priceExtension", "value", "oldPrice")
    ).useDiffer(new BigDecimalDiffer());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(AdsLogins.CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    static Long campaignId;
    static Long adGroupId;

    private static final Long ONE = 1000000L;
    private static final Long TEN = 10000000L;
    // Max value = 10^10 to micros
    private static final Long MAX = BigDecimal.TEN.pow(16).longValue();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public String description;
    @Parameterized.Parameter(1)
    public PriceExtensionUpdateItemMap price;
    @Parameterized.Parameter(2)
    public ExpectedResult expectedResult;

    private AdUpdateItemMap updateMap;
    private long bannerId;
    private PriceExtensionAddItemMap originalPrice;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        return Arrays.asList(
                new Object[]{"Цена равная нулю", price(0L, TEN, RUB, FROM),
                        lessThanMin("PriceExtension.Price", "0")},
                new Object[]{"Цена отрицательная", price(-ONE, TEN, RUB, FROM),
                        lessThanMin("PriceExtension.Price", "0")},
                new Object[]{"Старая цена отрицательная", price(ONE, -TEN, RUB, FROM),
                        lessThanMin("PriceExtension.OldPrice", "0")},
                new Object[]{"Старая цена меньше новой", price(TEN, ONE, RUB, FROM),
                        oldPriceLowerThanNew()},
                new Object[]{"Цена слишком большая", price(BigInteger.TEN.pow(20).longValue(), null, RUB, FROM),
                        moreThanMax("PriceExtension.Price", Long.toString(MAX))},
                new Object[]{"Старая цена слишком большая", price(TEN, BigInteger.TEN.pow(20).longValue(), RUB, FROM),
                        moreThanMax("PriceExtension.OldPrice", Long.toString(MAX))}
        );
    }

    private static ExpectedResult lessThanMin(String field, String value) {
        return errors(new Notification(5005, Api5ErrorDetailsJava.VALUE_MUST_BE_GREATER_THAN, field, value));
    }

    private static ExpectedResult moreThanMax(String field, String value) {
        return errors(new Notification(5005, Api5ErrorDetailsJava.VALUE_MUST_NOT_BE_GREATER_THAN, field, value));
    }

    private static ExpectedResult oldPriceLowerThanNew() {
        return errors(new Notification(5005, Api5ErrorDetailsJava.BANNER_OLD_PRICE_LOWER_THAN_NEW));
    }

    private static PriceExtensionUpdateItemMap price(Long price, Long oldPrice, PriceCurrencyEnum currency,
                                                     PriceQualifierEnum prefix)
    {
        return new PriceExtensionUpdateItemMap()
                .withPrice(price)
                .withOldPrice(oldPrice)
                .withPriceCurrency(currency)
                .withPriceQualifier(prefix);
    }

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdsLogins.CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(AdsLogins.CLIENT);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
    }

    @Before
    public void initMap() {
        originalPrice = new PriceExtensionAddItemMap().withPrice(ONE).withPriceCurrency(EUR).withPriceQualifier(NONE);
        AdAddItemMap addRequest = new AdAddItemMap()
                .withTextAd(new TextAdAddMap().defaultTextAd().withPriceExtension(originalPrice))
                .withAdGroupId(adGroupId);

        bannerId = api.userSteps.adsSteps().addAd(addRequest);

        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withIds(bannerId))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.TEXT)
                .withTextAdPriceExtensionFieldNames(TextAdPriceExtensionFieldEnum.values()));

        assumeThat("цена на товар на баннере создана верно", getResponse.getAds(),
                getMatcher(bannerId, originalPrice));

        updateMap = new AdUpdateItemMap()
                .withId(bannerId)
                .withTextAd(new TextAdUpdateMap().withPriceExtension(price));
    }

    @Test
    public void addWithBannerPriceAndCheckError() {
        //DIRECT-41067
        UpdateRequestMap request = new UpdateRequestMap().withAds(updateMap);
        api.userSteps.adsSteps().shouldGetResultOnUpdate(request, expectedResult);

        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withIds(bannerId))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.TEXT)
                .withTextAdPriceExtensionFieldNames(TextAdPriceExtensionFieldEnum.values()));
        assertThat("цена на товар на баннере не должна поменяться", getResponse.getAds(),
                getMatcher(bannerId, originalPrice));
    }
}
