package ru.yandex.autotests.direct.api.ads.update.textad;

import java.sql.Timestamp;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import javax.xml.bind.JAXBElement;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.AdGetItem;
import com.yandex.direct.api.v5.ads.GetResponse;
import com.yandex.direct.api.v5.ads.ObjectFactory;
import com.yandex.direct.api.v5.ads.PriceCurrencyEnum;
import com.yandex.direct.api.v5.ads.PriceExtensionUpdateItem;
import com.yandex.direct.api.v5.ads.PriceQualifierEnum;
import com.yandex.direct.api.v5.ads.TextAdFieldEnum;
import com.yandex.direct.api.v5.ads.TextAdPriceExtensionFieldEnum;
import com.yandex.direct.api.v5.ads.UpdateResponse;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.steps.BannersSteps;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdGetItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.PriceExtensionAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.PriceExtensionGetItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.PriceExtensionUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdGetMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.banners.AdsSteps;
import ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static com.yandex.direct.api.v5.ads.PriceCurrencyEnum.EUR;
import static com.yandex.direct.api.v5.ads.PriceCurrencyEnum.RUB;
import static com.yandex.direct.api.v5.ads.PriceQualifierEnum.FROM;
import static com.yandex.direct.api.v5.ads.PriceQualifierEnum.NONE;
import static java.util.Collections.singletonList;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Description("Проверка работы поля BannerPrice: позитивные кейсы")
@Issue("https://st.yandex-team.ru/DIRECT-89499")
@RunWith(Parameterized.class)
public class UpdateBannerPricePositiveTest {
    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(AdsLogins.CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    static Long campaignId;
    static Long adGroupId;

    private static final Long ONE = 1000000L;
    private static final Long TEN = 10000000L;

    @Rule
    public Trashman trashman = new Trashman(api);

    @Parameterized.Parameter(0)
    public String description;
    @Parameterized.Parameter(1)
    public PriceExtensionUpdateItemMap price;

    private AdUpdateItemMap updateMap;
    private Long bannerId;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        return Arrays.asList(
                new Object[]{"Заполненная цена", price(ONE, TEN, RUB, FROM)},
                new Object[]{"Цена с префиксом NONE", price(ONE, TEN, RUB, NONE)},
                new Object[]{"Цена без старой цены", price(ONE, null, RUB, FROM)},
                new Object[]{"Цена удалена", null});
    }

    private static PriceExtensionUpdateItemMap price(Long price, Long oldPrice, PriceCurrencyEnum currency, PriceQualifierEnum prefix) {
        return new PriceExtensionUpdateItemMap()
                .withPrice(price)
                .withOldPrice(oldPrice)
                .withPriceCurrency(currency)
                .withPriceQualifier(prefix);
    }

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdsLogins.CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(AdsLogins.CLIENT);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
    }

    @Before
    public void initMap() {
        PriceExtensionAddItemMap originalPrice = new PriceExtensionAddItemMap()
                .withPrice(ONE).withPriceCurrency(EUR).withPriceQualifier(FROM);
        AdAddItemMap addRequest = new AdAddItemMap()
                .withTextAd(new TextAdAddMap().defaultTextAd().withPriceExtension(originalPrice))
                .withAdGroupId(adGroupId);

        bannerId = api.userSteps.adsSteps().addAd(addRequest);

        GetResponse getResponse = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withIds(bannerId))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.TEXT)
                .withTextAdPriceExtensionFieldNames(TextAdPriceExtensionFieldEnum.values()));

        assumeThat("цена на товар на баннере создана верно", getResponse.getAds(),
                getMatcher(bannerId, originalPrice));

        TextAdUpdateMap textAdUpdateMap = new TextAdUpdateMap().withPriceExtension(price);
        if (price == null) {
            JAXBElement<PriceExtensionUpdateItem> priceExtension = new ObjectFactory()
                    .createTextAdUpdatePriceExtension(null);
            priceExtension.setNil(true);
            textAdUpdateMap.put(TextAdUpdateMap.PRICE_EXTENSION, priceExtension);
        }
        updateMap = new AdUpdateItemMap()
                .withId(bannerId)
                .withTextAd(textAdUpdateMap);
    }

    @Test
    public void addWithBannerPriceAndCheckSave() {
        //DIRECT-41067
        AdsSteps adsSteps = api.userSteps.adsSteps();
        BannersSteps bannerJooqSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(AdsLogins.CLIENT).bannersSteps();

        Timestamp timestampBeforeChange = Timestamp.from(Instant.now().minus(Duration.ofHours(1)));
        api.userSteps.bannersFakeSteps().setLastChange(bannerId, timestampBeforeChange.toString());

        UpdateResponse response = adsSteps.shouldGetResultOnUpdate(new UpdateRequestMap().withAds(updateMap),
                ExpectedResult.success());
        Long id = response.getUpdateResults().get(0).getId();
        GetResponse getResponse = adsSteps.adsGet(new GetRequestMap()
                .withSelectionCriteria(new AdsSelectionCriteriaMap().withIds(id))
                .withFieldNames(AdFieldEnum.ID)
                .withTextAdFieldNames(TextAdFieldEnum.TEXT)
                .withTextAdPriceExtensionFieldNames(TextAdPriceExtensionFieldEnum.values()));

        assertThat("Время изменений должно обновиться",
                bannerJooqSteps.getBanner(bannerId).getLastchange(), not(equalTo(timestampBeforeChange)));
        if (price != null) {
            assertThat("цена на товар на баннере сохранась верно", getResponse.getAds(),
                    getMatcher(id, price));
        } else {
            assertThat("цена должна быть удалена", getResponse.getAds().get(0).getTextAd()
                    .getPriceExtension().getValue(), nullValue());
        }
    }

    static BeanDifferMatcher<List<AdGetItem>> getMatcher(long id, PriceExtensionAddItemMap price) {
        return beanDifferV5(
                singletonList((AdGetItem) new AdGetItemMap().withId(id).withTextAd(
                        new TextAdGetMap()
                                .withText(TextAdAddMap.DEFAULT_TEXT)
                                .withPriceExtension(convertAddToGet(price))).getBean()));
    }

    static BeanDifferMatcher<List<AdGetItem>> getMatcher(long id, PriceExtensionUpdateItemMap price) {
        return beanDifferV5(
                singletonList((AdGetItem) new AdGetItemMap().withId(id).withTextAd(
                        new TextAdGetMap()
                                .withText(TextAdAddMap.DEFAULT_TEXT)
                                .withPriceExtension(convertUpdateToGet(price))).getBean()));
    }

    private static PriceExtensionGetItemMap convertAddToGet(PriceExtensionAddItemMap add) {
        return new PriceExtensionGetItemMap()
                .withPrice((Long) add.get(PriceExtensionAddItemMap.PRICE))
                .withOldPrice((Long) add.get(PriceExtensionAddItemMap.OLD_PRICE))
                .withPriceCurrency((PriceCurrencyEnum) add.get(PriceExtensionAddItemMap.PRICE_CURRENCY))
                .withPriceQualifier((PriceQualifierEnum) add.get(PriceExtensionAddItemMap.PRICE_QUALIFIER));
    }

    private static PriceExtensionGetItemMap convertUpdateToGet(PriceExtensionUpdateItemMap update) {
        return new PriceExtensionGetItemMap()
                .withPrice((Long) update.get(PriceExtensionAddItemMap.PRICE))
                .withOldPrice(((JAXBElement<Long>) update.get(PriceExtensionAddItemMap.OLD_PRICE)).getValue())
                .withPriceCurrency((PriceCurrencyEnum) update.get(PriceExtensionAddItemMap.PRICE_CURRENCY))
                .withPriceQualifier((PriceQualifierEnum) update.get(PriceExtensionAddItemMap.PRICE_QUALIFIER));
    }
}
