package ru.yandex.autotests.direct.api.ads.update.textad;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.ads.AddResponse;
import org.hamcrest.Matcher;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannerDisplayHrefsStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannerDisplayHrefsRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by ginger on 07.04.16.
 * https://st.yandex-team.ru/TESTIRT-8931
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Description("Проверка BsSync и LastChange после добавления, обновления, удаления поля DisplayUrlPath.")
@Issue("https://st.yandex-team.ru/DIRECT-52434")
@RunWith(Parameterized.class)
public class UpdateDisplayUrlPathBsSyncAndLastChangeTest {
    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(AdsLogins.CLIENT);
    private static DirectJooqDbSteps jooqDbSteps;

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    static Long campaignId;
    static Long adGroupId;
    TextAdAddMap textAdAddMap;
    private Long adId;
    private BannerFakeInfo actualBannerFakeInfo;
    private static DateTime lastChange = DateTime.now().minusDays(1);
    private static final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public String pathForAdd;

    @Parameterized.Parameter(2)
    public String pathForUpdate;

    @Parameterized.Parameter(3)
    public String expectedStatusBsSync;

    @Parameterized.Parameter(4)
    public Matcher expectedDisplayHrefsStatus;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        BannerDisplayHrefsRecord expectedBannerDisplayHrefs = new BannerDisplayHrefsRecord();
        expectedBannerDisplayHrefs.setStatusmoderate(BannerDisplayHrefsStatusmoderate.Ready);

        return Arrays.asList(new Object[][]{
                {"Добавление", null, TextAdAddMap.DEFAULT_DISPLAY_URL_PATH, Status.YES, equalTo(BannerDisplayHrefsStatusmoderate.Ready)},
                {"Удаление", TextAdAddMap.DEFAULT_DISPLAY_URL_PATH, null, Status.NO, nullValue()},
                {"Редактирование", TextAdAddMap.DEFAULT_DISPLAY_URL_PATH,
                        "new"+TextAdAddMap.DEFAULT_DISPLAY_URL_PATH, Status.YES, equalTo(BannerDisplayHrefsStatusmoderate.Ready)}
        });
    }

    @BeforeClass
    public static void initTest() {
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(AdsLogins.CLIENT);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdsLogins.CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        jooqDbSteps.useShardForLogin(AdsLogins.CLIENT)
                .campaignsSteps().setCampaignsStatusModerate(campaignId, CampaignsStatusmoderate.Yes);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
    }

    @Before
    public void initMap() {
        textAdAddMap = new TextAdAddMap().defaultTextAd().withDisplayUrlPath(pathForAdd);

        AddResponse response = api.userSteps.adsSteps().shouldGetResultOnAdd(new AddRequestMap()
                .withAds(new AdAddItemMap()
                        .withTextAd(
                                new TextAdAddMap()
                                        .defaultTextAd()
                                        .withDisplayUrlPath(pathForAdd)
                        )
                        .withAdGroupId(adGroupId)), ExpectedResult.success());
        adId = response.getAddResults().get(0).getId();

        api.userSteps.groupFakeSteps().makeGroupFullyModerated(adGroupId);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(adId);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adId, Status.YES);
        api.userSteps.bannersFakeSteps().setLastChange(adId, lastChange.toString());


        api.userSteps.adsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(adId)
                                .withTextAd(
                                        new TextAdUpdateMap().withDisplayUrlPath(pathForUpdate))),
                ExpectedResult.success());

        actualBannerFakeInfo = api.userSteps.bannersFakeSteps().getBannerParams(adId);
    }

    @Test
    @Issue("DIRECT-53049")
    public void checkBsSync() {
        assertThat("статусы объявления обновлись верно",
                actualBannerFakeInfo.getStatusBsSynced(), equalTo(expectedStatusBsSync));
    }

    @Test
    public void checkLastChanges() {
        LocalDate actualDate = formatter.parseLocalDate(actualBannerFakeInfo.getLastChange());
        assertThat("LastChange объявления обновился", actualDate, equalTo(LocalDate.now()));
    }

    @Test
    public void checkDisplayHrefsStatus() {
        BannerDisplayHrefsRecord bannerDisplayHrefs = jooqDbSteps.bannerDisplayHrefsSteps().getBannerDisplayHrefs(adId);
        BannerDisplayHrefsStatusmoderate displayHrefsStatus = bannerDisplayHrefs != null ? bannerDisplayHrefs.getStatusmoderate() : null;
        assertThat("статус ссылки обновлся верно", displayHrefsStatus, expectedDisplayHrefsStatus);
    }
}
