package ru.yandex.autotests.direct.api.ads.update.textad;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.ads.AdsFeatures;
import ru.yandex.autotests.direct.api.ads.AdsLogins;
import ru.yandex.autotests.direct.api.ads.AdsStories;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.sitelinks.SitelinkMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.RandomStringUtils.randomAlphabetic;

/**
 * Created by semkagtn on 13.04.15.
 * https://st.yandex-team.ru/TESTIRT-5113
 */
@Aqua.Test
@Features(AdsFeatures.UPDATE)
@Stories(AdsStories.TEXT_AD)
@Description("Невалидные ссылки в запросе на обновлении текстового объявления")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class UpdateHrefNegativeTest {

    private static final String LOGIN = AdsLogins.CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public String href;

    @Parameterized.Parameter(2)
    public Notification expectedNotification;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        String wrongProtocolHref = "ssh://semkagtn.com";
        String onlyNumericHref = "http://1234.56";
        String hrefWithSpecificSymbol = "http://hello!world.ru";
        String lessThanMinDomainLengthHref = "http://" + randomAlphabetic(SitelinkMap.MIN_DOMAIN_LENGTH - 1) + ".ru";
        String greaterThanMaxDomainLengthHref = "http://" + randomAlphabetic(SitelinkMap.MAX_DOMAIN_LENGTH + 1) +
                "." + randomAlphabetic(SitelinkMap.MAX_DOMAIN_LENGTH) + ".ru";
        String lessThanMinTopDomainLengthHref =
                "http://aaaaa." + randomAlphabetic(SitelinkMap.MIN_TOP_DOMAIN_LENGTH - 1);
        String greaterThanMaxTopDomainLengthHref =
                "http://tester." + randomAlphabetic(SitelinkMap.MAX_TOP_DOMAIN_LENGTH + 1);
        String greaterThanMaxLengthHref =
                "http://" + generateValidHrefWithoutProtocolOfLength(SitelinkMap.MAX_HREF_LENGTH - 6);
        String emptyHref = "";

        return Arrays.asList(new Object[][]{
                {"Обновить объявление со ссылкой, протокол которой отличен от http или https", wrongProtocolHref,
                        new Notification(5005, Api5ErrorDetails.INCORRECT_LINK, wrongProtocolHref)},

                {"Обновить объявление со ссылкой, состоящей только из цифр", onlyNumericHref,
                        new Notification(5005, Api5ErrorDetails.INCORRECT_LINK, onlyNumericHref)},

                {"Обновить объявление со ссылкой, содержащей специфичный символ", hrefWithSpecificSymbol,
                        new Notification(5005, Api5ErrorDetails.INCORRECT_LINK, hrefWithSpecificSymbol)},

                {"Обновить объявление со ссылкой с внутренним доменом длиной меньшей, чем " +
                        TextAdAddMap.MIN_DOMAIN_LENGTH, lessThanMinDomainLengthHref, new Notification(5005,
                        Api5ErrorDetails.INCORRECT_LINK, lessThanMinDomainLengthHref)},

                {"Обновить объявление со ссылкой с внутренними доменами длиной большей, чем " +
                        TextAdAddMap.MAX_DOMAIN_LENGTH, greaterThanMaxDomainLengthHref, new Notification(5005,
                        Api5ErrorDetails.INCORRECT_LINK, greaterThanMaxDomainLengthHref)},

                {"Обновить объявление со ссылкой с доменом верхнего уровня длиной меньшей, чем" +
                        TextAdAddMap.MIN_TOP_DOMAIN_LENGTH, lessThanMinTopDomainLengthHref, new Notification(5005,
                        Api5ErrorDetails.INCORRECT_LINK, lessThanMinTopDomainLengthHref)},

                {"Обновить объявление со ссылкой с доменом верхнего уровня длиной большей, чем" +
                        TextAdAddMap.MAX_TOP_DOMAIN_LENGTH, greaterThanMaxTopDomainLengthHref, new Notification(5005,
                        Api5ErrorDetails.INCORRECT_LINK, greaterThanMaxTopDomainLengthHref)},

                {"Обновить объявление со ссылкой, длиной большей, чем " + TextAdAddMap.MAX_HREF_LENGTH,
                        greaterThanMaxLengthHref, new Notification(5005,
                        Api5ErrorDetails.LINK_HAS_EXCEEDED_CHARACTER_LIMIT, TextAdAddMap.MAX_HREF_LENGTH)},

                {"Обновить объявление с пустой ссылкой", emptyHref,
                        new Notification(5003, Api5ErrorDetailsJava.EMPTY_HREF)}
        });
    }

    private static long adID;

    @BeforeClass
    public static void createAd() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        adID = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
    }

    @Before
    public void addUnitsToClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    public void updateHref() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(adID)
                                .withTextAd(new TextAdUpdateMap()
                                        .withHref(href))),
                ExpectedResult.errors(expectedNotification));
    }

    private static String generateValidHrefWithoutProtocolOfLength(int length) {
        StringBuilder stringBuilder = new StringBuilder(randomAlphabetic(length));
        int domainLength = Math.min(
                (SitelinkMap.MAX_DOMAIN_LENGTH - 1) / 2, (SitelinkMap.MAX_TOP_DOMAIN_LENGTH - 1) / 2);
        for (int i = domainLength; i < length - domainLength; i += domainLength) {
            stringBuilder.setCharAt(i, '.');
        }
        return stringBuilder.toString();
    }
}
